/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.seam.solder.beanManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.Extension;
import javax.inject.Inject;

import org.jboss.seam.solder.util.Sortable;
import org.jboss.seam.solder.util.service.ServiceLoader;

/**
 * <p>
 * A utility for use in non-managed classes, which are not able to obtain a
 * reference to the {@link BeanManager} using injection.
 * </p>
 * 
 * <p>
 * {@link BeanManagerProvider} is an SPI that may be implemented to allow third
 * parties to register custom methods of looking up the BeanManager in an
 * external context. This class will consult implementations according to
 * precedence.
 * </p>
 * 
 * <p>
 * <b>**WARNING**</b> This class is <b>NOT</b> a clever way to get the
 * BeanManager, and should be <b>avoided at all costs</b>. If you need a handle
 * to the {@link BeanManager} you should probably register an {@link Extension}
 * instead of using this class; have you tried using @{@link Inject}?
 * </p>
 * 
 * <p>
 * If you think you need to use this class, chat to the community and make sure
 * you aren't missing a trick!
 * </p>
 * 
 * @see BeanManagerProvider
 * @see BeanManagerAware
 * 
 * @author Pete Muir
 * @author Nicklas Karlsson
 * @author <a href="http://community.jboss.org/people/dan.j.allen">Dan Allen</a>
 */
public class BeanManagerLocator
{
   private BeanManager beanManager;
   
   private boolean lookupPerformed = false;
   
   private BeanManagerProvider locatingProvider;

   private List<BeanManagerProvider> providers;

   /**
    * If a lookup has not yet been performed, consult registered
    * {@link BeanManagerProvider} implementations to locate the
    * {@link BeanManager} and return the result. If the {@link BeanManager}
    * cannot be resolved, throw a {@link BeanManagerUnavailableException}.
    * 
    * @throws BeanManagerUnavailableException if the BeanManager cannot be resolved
    * @return the BeanManager for the current bean archive
    */
   public BeanManager getBeanManager()
   {
      if (!lookupPerformed)
      {
         beanManager = lookupBeanManager();
      }
      
      if (beanManager == null)
      {
         throw new BeanManagerUnavailableException(providers);
      }
      
      return beanManager;
   }

   /**
    * If a lookup has not yet been performed, consult registered
    * {@link BeanManagerProvider} implementations to locate the
    * {@link BeanManager} and return whether it was found, caching
    * the result.
    * 
    * @return <code>true</code> if the bean manager has been found, otherwise
    *         <code>false</code>
    */
   public boolean isBeanManagerAvailable()
   {
      if (!lookupPerformed)
      {
         beanManager = lookupBeanManager();
      }
      
      return beanManager != null;
   }
   
   /**
    * Return the {@link BeanManagerProvider} that was used to
    * locate the BeanManager. This method will not attempt
    * a lookup.
    * 
    * @return the BeanManagerProvider implementation
    */
   public BeanManagerProvider getLocatingProvider()
   {
      return locatingProvider;
   }

   private BeanManager lookupBeanManager()
   {
      BeanManager beanManager = null;
      if (providers == null)
      {
         loadServices();
         Collections.sort(providers, new Sortable.Comparator());
      }
      
      for (BeanManagerProvider provider : providers)
      {
         beanManager = provider.getBeanManager();
         if (beanManager != null)
         {
            locatingProvider = provider;
            break;
         }
      }
      
      lookupPerformed = true;
      return beanManager;
   }
   
   private void loadServices()
   {
      providers = new ArrayList<BeanManagerProvider>();
      
      for (Iterator<BeanManagerProvider> it = ServiceLoader.load(BeanManagerProvider.class).iterator(); it.hasNext();)
      {
         providers.add(it.next());
      }
   }
}
