/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.acl;

import org.jboss.security.authorization.Resource;

/**
 * <p>
 * An <code>ACLProvider</code> is responsible for the management of the ACLs associated to the
 * resources being protected. Implementations of this interface will tipically interact with some
 * sort of repository, where the ACLs are stored.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public interface ACLProvider
{

   /**
    * <p>
    * Obtains a reference to the <code>ACL</code> associated to the given resource.
    * </p>
    * 
    * @param resource   the <code>Resource</code> for which the associated ACL is wanted.
    * @return   a reference to the <code>ACL</code> associated with the resource, or null if no
    * ACL could be found.
    */
   public ACL retrieveACL(Resource resource);

   /**
    * <p>
    * Creates a new ACL and associates it to the given resource.
    * </p>
    * 
    * @param resource   the <code>Resource</code> for which an ACL is to be created.
    * @return   a reference to the created <code>ACL</code>.
    */
   public ACL createACL(Resource resource);

   /**
    * <p>
    * Updates the given ACL. This usually means updating the repository where the ACLs are stored.
    * </p>
    * 
    * @param acl the <code>ACL</code> that needs to be updated.
    * @return   <code>true</code> if the ACL was updated; <code>false</code> otherwise.
    */
   public boolean updateACL(ACL acl);

   /**
    * <p>
    * Removes the given ACL, breaking the existing association with the resource it relates to.
    * </p>
    * 
    * @param acl    a reference to the <code>ACL</code> that is to be removed.
    * @return   <code>true</code> if the ACL was removed; <code>false</code> otherwise.
    */
   public boolean removeACL(ACL acl);

   /**
    * <p>
    * Removes the ACL associated with the specified resource.
    * </p>
    * 
    * @param resource    the <code>Resource</code> whose associated ACL is to be removed.
    * @return   <code>true</code> if the ACL was removed; <code>false</code> otherwise.
    */
   public boolean removeACL(Resource resource);

}
