/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.AppConfigurationEntry.LoginModuleControlFlag;

import org.jboss.security.audit.config.AuditProviderEntry;
import org.jboss.security.auth.login.AuthenticationInfo;
import org.jboss.security.auth.login.BaseAuthenticationInfo;
import org.jboss.security.authorization.config.AuthorizationModuleEntry;
import org.jboss.security.config.jaxb.LoginModuleInfo;
import org.jboss.security.config.jaxb.ModuleOption;
import org.jboss.security.config.jaxb.ModuleOptionFlag;
import org.jboss.security.config.jaxb.PolicyModule;
import org.jboss.security.config.jaxb.ProviderModule;

/**
 * Bridges the JAXB Configuration to the JBossSX one
 * <a href="mailto:anil.saldhana@redhat.com>Anil Saldhana</a>
 * @since May 30, 2008
 */
public class JBossSXConfigBridge
{
   public ApplicationPolicy map(org.jboss.security.config.jaxb.ApplicationPolicy appPolicy)
   {
     String appName = appPolicy.getName();
     ApplicationPolicy result = new ApplicationPolicy(appName);
     result.setAuthenticationInfo(map(appName, appPolicy.getAuthentication()));
     result.setAuthorizationInfo(map(appName, appPolicy.getAuthorization()));
     result.setAuditInfo(map(appName, appPolicy.getAudit()));
     return result;
   } 
   
   public BaseAuthenticationInfo map(String name, 
         org.jboss.security.config.jaxb.AuthenticationInfo authenticationInfo)
   {
     AuthenticationInfo baseInfo = new AuthenticationInfo(name);  
     baseInfo.setAppConfigurationEntry(map(authenticationInfo.getLoginModule()));
     return baseInfo;
   }
   
   public AuthorizationInfo map(String name, 
         org.jboss.security.config.jaxb.AuthorizationInfo authorizationInfo)
   {
     if(authorizationInfo == null)
        return null;
     AuthorizationInfo authzInfo = new AuthorizationInfo(name);
     List<PolicyModule> policyModuleList = authorizationInfo.getPolicyModule();
     int lenOfPolicyModules = policyModuleList != null ? policyModuleList.size() : 0;
     if(lenOfPolicyModules > 0)
     {
        for(PolicyModule policyModule : policyModuleList)
        {
           AuthorizationModuleEntry authzEntry = new AuthorizationModuleEntry(policyModule.getCode());
           authzInfo.add(authzEntry);
           authzEntry.setControlFlag(getControlFlag(policyModule.getFlag()));  
           authzEntry.getOptions().putAll(map(policyModule.getModuleOption())); 
        }
     }
     return authzInfo;
   }
   
   public AuditInfo map(String name, org.jboss.security.config.jaxb.AuditInfo auditInfo)
   {
      if(auditInfo == null)
         return null;
      AuditInfo audInfo = new AuditInfo(name);
      audInfo.add(map(auditInfo.getProviderModule()));
      return audInfo;
   }
   
   public AppConfigurationEntry[] 
               map(List<org.jboss.security.config.jaxb.LoginModuleInfo> loginModuleInfoList)
   {
      int index = 0;
      AppConfigurationEntry[] entries = new AppConfigurationEntry[loginModuleInfoList.size()];
      
      for(LoginModuleInfo loginModuleInfo : loginModuleInfoList)
      {
         entries[index++] = map(loginModuleInfo);
      }
      return entries;
   }
   
   public AppConfigurationEntry map(org.jboss.security.config.jaxb.LoginModuleInfo loginModuleInfo)
   {
      String moduleName = loginModuleInfo.getCode();
      ModuleOptionFlag flag = loginModuleInfo.getFlag();
      List<ModuleOption> moduleOptionList = loginModuleInfo.getModuleOption(); 
      return new AppConfigurationEntry(moduleName, map(flag), map(moduleOptionList)); 
   }
   
   public Map<String,Object> map(List<ModuleOption> moduleOptionList)
   {
      if(moduleOptionList == null)
         return null;
      Map<String,Object> moduleOptions = new HashMap<String,Object>();
      for(ModuleOption moduleOption: moduleOptionList)
      {
         String name = moduleOption.getName();
         List<Object> objectList = moduleOption.getContent();
         moduleOptions.put(name, objectList.get(0)); //Just get the first object 
      }
      return moduleOptions;
   }
   
   public List<AuditProviderEntry> 
        map(List<org.jboss.security.config.jaxb.ProviderModule> providerModulesList)
   {
      List<AuditProviderEntry> auditProviderEntries = new ArrayList<AuditProviderEntry>();
      
      int len = providerModulesList != null ? providerModulesList.size() : 0;
      if(len > 0)
      {
         for(ProviderModule providerModule : providerModulesList)
         {
            AuditProviderEntry ape = new AuditProviderEntry(providerModule.getCode());  
            ape.getOptions().putAll(map(providerModule.getModuleOption()));
            auditProviderEntries.add(ape);
         }
      }
      return auditProviderEntries;
   }

   public LoginModuleControlFlag map(ModuleOptionFlag flag)
   {
      if(flag != null)
      {
         if(flag.compareTo(ModuleOptionFlag.OPTIONAL) == 0)
            return LoginModuleControlFlag.OPTIONAL;
         else
         if(flag.compareTo(ModuleOptionFlag.SUFFICIENT) == 0)
            return LoginModuleControlFlag.SUFFICIENT;
         else
         if(flag.compareTo(ModuleOptionFlag.REQUISITE) == 0)
            return LoginModuleControlFlag.REQUISITE;
      }
      
      //Get the default option
      return LoginModuleControlFlag.REQUIRED;      
   }
   
   public ControlFlag getControlFlag(ModuleOptionFlag flag)
   {
      if(flag != null)
      {
         if(flag.compareTo(ModuleOptionFlag.OPTIONAL) == 0)
            return ControlFlag.OPTIONAL;
         else
         if(flag.compareTo(ModuleOptionFlag.SUFFICIENT) == 0)
            return ControlFlag.SUFFICIENT;
         else
         if(flag.compareTo(ModuleOptionFlag.REQUISITE) == 0)
            return ControlFlag.REQUISITE;
      }
      
      //Get the default option
      return ControlFlag.REQUIRED;
   }
}