/*
  * JBoss, Home of Professional Open Source
  * Copyright 2007, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.plugins;
 
import java.security.Principal;
import java.util.HashMap;
import java.util.Map;

import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException; 
import javax.security.auth.message.AuthException;
import javax.security.auth.message.AuthStatus;
import javax.security.auth.message.MessageInfo;
import javax.security.auth.message.config.AuthConfigFactory;
import javax.security.auth.message.config.AuthConfigProvider;
import javax.security.auth.message.config.ServerAuthConfig;
import javax.security.auth.message.config.ServerAuthContext;
import javax.security.jacc.PolicyContext;

import org.jboss.logging.Logger;
import org.jboss.security.AuthenticationManager;
import org.jboss.security.SecurityConstants;
import org.jboss.security.auth.callback.AppCallbackHandler;
import org.jboss.security.cache.JBossAuthenticationCache;
import org.jboss.security.cache.SecurityCache;
import org.jboss.security.cache.SecurityCacheException;

//$Id$

/**
 *  Default Implementation of the AuthenticationManager Interface
 *  @author Anil.Saldhana@redhat.com
 *  @since  May 10, 2007 
 *  @version $Revision$
 */
public class JBossAuthenticationManager implements AuthenticationManager
{
   private static Logger log = Logger.getLogger(JBossAuthenticationManager.class);
   
   protected String securityDomain = SecurityConstants.DEFAULT_APPLICATION_POLICY;
   
   protected CallbackHandler callbackHandler = null;
    
   private ThreadLocal<Subject> subjectLocal = new ThreadLocal<Subject>();
   
   private SecurityCache<Principal> sCache = null;

   private boolean cacheValidation = false; 
    
   public JBossAuthenticationManager(String sdomain, CallbackHandler cbh)
   {
      this.securityDomain = sdomain;
      this.callbackHandler = cbh;
      sCache = new JBossAuthenticationCache();
   }
   
   /**
    * Create JBossAuthenticationManager
    * @param sdomain SecurityDomain
    * @param cbh CallbackHandler
    * @param initCapacity Initial Capacity for the internal Security Cache
    * @param loadFactor Load Factor for the internal Security Cache
    * @param level Concurrency Level for the internal Security Cach
    */
   public JBossAuthenticationManager(String sdomain, CallbackHandler cbh, 
         int initCapacity, float loadFactor, int level)
   {
      this.securityDomain = sdomain;
      this.callbackHandler = cbh;
      sCache = new JBossAuthenticationCache(initCapacity, loadFactor, level);
   }
   
   public void setSecurityCache(String className)
   {
      if(className == null)
         throw new IllegalArgumentException("className is null");
      ClassLoader cl = SubjectActions.getContextClassLoader();
      try
      {
         Class clazz = cl.loadClass(className);
         sCache = (SecurityCache<Principal>) clazz.getConstructor(new Class[]{}).newInstance(new Object[]{});
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }

   /**
    * @see AuthenticationManager#getActiveSubject()
    */
   public Subject getActiveSubject()
   { 
      return subjectLocal.get();
   }

   /**
    * @see AuthenticationManager#getSecurityDomain()
    */
   public String getSecurityDomain()
   {
      return securityDomain;
   }

   /**
    * @see AuthenticationManager#getTargetPrincipal(Principal, Map)
    */
   public Principal getTargetPrincipal(Principal principal, Map<String,Object> map)
   {
      throw new RuntimeException("Unimplemented");
   }

   /**
    * @see AuthenticationManager#isValid(Principal, Object)
    */
   public boolean isValid(Principal principal, Object credential)
   {
      return isValid(principal, credential, new Subject());
   } 

   /**
    * @see AuthenticationManager#isValid(Principal, Object, Subject)
    */
   public boolean isValid(Principal principal, Object credential, Subject subject)
   {
      if(subject == null)
         throw new IllegalArgumentException("Subject is null"); 

      HashMap<String,Object> map = new HashMap<String,Object>();
      if(sCache.cacheHit(principal))
      {
         Subject cacheSubject = validateCache(principal,credential,subject);
         if(cacheSubject != null)
         {
            subject = cacheSubject;
            subjectLocal.set(cacheSubject);
            return true;
         } 
      }
      LoginContext lc = null;
      
      try
      {
         this.cacheValidation = false;
         lc = new LoginContext(securityDomain, subject, callbackHandler);
         lc.login(); 

         map.put(SecurityConstants.CREDENTIAL, credential);
         map.put(SecurityConstants.SUBJECT, subject);
         try
         {
            sCache.addCacheEntry(principal, map);
         }
         catch (SecurityCacheException e)
         {
            throw new RuntimeException(e);
         }
         subjectLocal.set(lc.getSubject());
      }
      catch (LoginException e)
      {
         log.trace("Login Failure:",e);
         return false;
      }
      return true;
   } 
   
   /**
    * @see AuthenticationManager#isValid(MessageInfo, Subject, String)
    */
   public boolean isValid(MessageInfo requestMessage,Subject clientSubject, String layer)
   { 
      AuthStatus status = AuthStatus.FAILURE;
      
      try
      {
         String contextID = PolicyContext.getContextID();
         AuthConfigFactory factory = AuthConfigFactory.getFactory();
         AuthConfigProvider provider = factory.getConfigProvider(layer,contextID,null); 
         ServerAuthConfig serverConfig = provider.getServerAuthConfig(layer,contextID, 
                  new AppCallbackHandler("DUMMY","DUMMY".toCharArray()));  
         ServerAuthContext sctx = serverConfig.getAuthContext(contextID, 
               new Subject(), new HashMap());
         if(clientSubject == null)
            clientSubject = new Subject();
         Subject serviceSubject = new Subject();
         status = sctx.validateRequest(requestMessage, clientSubject, serviceSubject); 
         this.subjectLocal.set(clientSubject);
      }
      catch(AuthException ae)
      {
         log.trace("AuthException:",ae);
      } 
      return AuthStatus.SUCCESS == status ;
   }

   /**
    * Value added method for testing alone
    * @return
    */
   public boolean fromCache()
   {
      return cacheValidation ;
   }
   
   private Subject validateCache(Principal principal, Object credential, Subject subject)
   {
      this.cacheValidation = false;
      HashMap<String,Object> map = new HashMap<String,Object>();
      map.put(SecurityConstants.CREDENTIAL, credential);
      try
      {
         sCache.cacheOperation(principal, map);
         Object cacheReturn = sCache.get(principal); 
         if(cacheReturn != null && cacheReturn instanceof Subject)
         {
            subject = (Subject) cacheReturn;
            this.cacheValidation = true;
            return subject; 
         } 
      }
      catch (SecurityCacheException e)
      { 
      }
      return null;
   } 
}
