/*
  * JBoss, Home of Professional Open Source.
  * Copyright 2006, Red Hat Middleware LLC, and individual contributors
  * as indicated by the @author tags. See the copyright.txt file in the
  * distribution for a full listing of individual contributors. 
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.mapping.providers;

import java.io.IOException;
import java.security.Principal;
import java.security.acl.Group;
import java.util.ArrayList;
import java.util.Enumeration; 
import java.util.Map;
import java.util.Properties; 
import java.util.Set;

import javax.security.jacc.PolicyContext;

import org.jboss.logging.Logger;
import org.jboss.security.SecurityConstants;
import org.jboss.security.mapping.MappingProvider;
import org.jboss.security.mapping.MappingResult;

//$Id$

/**
 *  A Role Mapping Provider that provider role mapping at the policy context id
 *  level (basically at deployment unit levels- war, ear etc)
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Oct 10, 2006 
 *  @version $Revision$
 */
public class PolicyContextIdRoleMappingProvider implements MappingProvider<Group>
{ 
   private static Logger log = Logger.getLogger(PolicyContextIdRoleMappingProvider.class);
   private boolean trace = log.isTraceEnabled();
   
   private Map options = null; 
   private MappingResult<Group> result;
   
   private Properties policyContextIdToFileNameProps = new Properties();
   
   private static final String PROPERTIES_ID = "map"; 
   private static final String EXTERNAL_ID = "externalProperties";

   public void init(Map opt)
   { 
      this.options = opt;
      if(trace)
         log.trace("Module Options=" + opt);
      //Check if there is inline properties
      if(options != null)
      {
         //Load up the PolicyContextId to properties-file-name properties
         policyContextIdToFileNameProps = (Properties)options.get(PROPERTIES_ID); 
         if(policyContextIdToFileNameProps == null)
         {
            //Check if was a properties file specified
            String filename = (String)options.get(EXTERNAL_ID);
            if(filename != null)
            try
            {
               policyContextIdToFileNameProps = loadProperties(filename);
            }
            catch (IOException e)
            {
               if(trace)
                  log.trace("Loading external properties file=", e);
            }
         }
      }
   }

   public void setMappingResult(MappingResult res)
   { 
      result = res;
   }
   
   public void performMapping(Map contextMap, Group mappedObject)
   { 
      ArrayList <Principal> tempPrincipalList = new ArrayList<Principal>(); 
      
      Principal principal = null;
      Set<Principal> principals = (Set)contextMap.get(SecurityConstants.PRINCIPALS_SET_IDENTIFIER);
      for(Principal p:principals)
      {
         if(p instanceof Group == false)
         {
            principal = p;
            break;
         }
      }  
      
      if(principal == null)
         throw new IllegalStateException("Authenticated Principal not found");
      
      //Get the current policy context id
      String contextId = PolicyContext.getContextID();
      if(trace)
         log.trace("Policy Context ID="+ contextId);
      if(contextId != null && policyContextIdToFileNameProps != null 
            && policyContextIdToFileNameProps.containsKey(contextId))
      {
        //check if the current policy context id is present in the map
        String filename = policyContextIdToFileNameProps.getProperty(contextId);
        try
        {
           Properties props = loadProperties(filename);
           String rolesToAdd = props.getProperty(principal.getName());
           String[] addRoles = null;
           if(rolesToAdd != null)
              addRoles = MappingProviderUtil.getRolesFromCommaSeparatedString(rolesToAdd); 
           
           //Create a temp list of principals stored in the role group
           Enumeration<? extends Principal> groupMembers = mappedObject.members();
           while(groupMembers.hasMoreElements())
           {
              tempPrincipalList.add(groupMembers.nextElement());
           }
           
           //Remove all the previous roles from the Group
           for(Principal p:tempPrincipalList)
           {
              mappedObject.removeMember(p);
           }
           
           //Now add the new roles to the group
           MappingProviderUtil.addRoles(mappedObject, addRoles);
        }
        catch (IOException e)
        {
           if(trace)
              log.trace("Exception:",e);
        }
      }   
      result.setMappedObject(mappedObject);
   } 
   
   private Properties loadProperties(String filename) throws IOException
   {
      Properties props = new Properties();
      ClassLoader tcl = Thread.currentThread().getContextClassLoader();
      props.load(tcl.getResourceAsStream(filename));
      return props;
   }
}