/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.acl;

import java.util.HashMap;
import java.util.Map;

import org.jboss.security.authorization.Resource;

/**
 * <p>
 * This class is a simple <code>ACLProvider</code> implementation that maintains the ACLs in memory. It is
 * used mainly for testing purposes.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class ACLProviderImpl implements ACLProvider
{

   // in memory cache of the created ACLs.
   private final Map<Resource, ACL> aclMap;

   /**
    * <p>
    * Creates an instance of <code>ACLProviderImpl</code>.
    * </p>
    */
   public ACLProviderImpl()
   {
      this.aclMap = new HashMap<Resource, ACL>();
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLProvider#createACL(org.jboss.security.authorization.Resource)
    */
   public ACL createACL(Resource resource)
   {
      if (resource == null)
         throw new IllegalArgumentException("ACLs cannot be created for null resources");
      ACL acl = this.aclMap.get(resource);
      if (acl == null)
      {
         acl = new ACLImpl(resource);
         this.aclMap.put(resource, acl);
      }
      return acl;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLProvider#removeACL(org.jboss.security.acl.ACL)
    */
   public boolean removeACL(ACL acl)
   {
      return this.removeACL(acl.getResource());
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLProvider#removeACL(org.jboss.security.authorization.Resource)
    */
   public boolean removeACL(Resource resource)
   {
      return this.aclMap.remove(resource) != null;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLProvider#retrieveACL(org.jboss.security.authorization.Resource)
    */
   public ACL retrieveACL(Resource resource)
   {
      return this.aclMap.get(resource);
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACLProvider#updateACL(org.jboss.security.acl.ACL)
    */
   public boolean updateACL(ACL acl)
   {
      if (!this.aclMap.containsKey(acl.getResource()))
         return false;
      return this.aclMap.put(acl.getResource(), acl) != null;
   }

}
