/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.authorization.modules.ejb;

import java.io.ByteArrayOutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.Principal;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.security.authorization.XACMLConstants;
import org.jboss.security.identity.Role;
import org.jboss.security.identity.RoleGroup;
import org.jboss.security.identity.RoleType;

import com.sun.xacml.Indenter;
import com.sun.xacml.attr.StringAttribute;
import com.sun.xacml.attr.TimeAttribute;
import com.sun.xacml.ctx.Attribute;
import com.sun.xacml.ctx.RequestCtx;
import com.sun.xacml.ctx.Subject;

//$Id: EJBXACMLUtil.java 68749 2008-01-09 20:25:39Z anil.saldhana@jboss.com $

/**
 *  Utility class for the XACML Integration for the EJB Layer
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jul 6, 2006 
 *  @version $Revision: 68749 $
 */
public class EJBXACMLUtil
{
   private static Logger log = Logger.getLogger(EJBXACMLUtil.class);
   private boolean trace = log.isTraceEnabled();
   
   public EJBXACMLUtil()
   {   
   }
   
   public RequestCtx createXACMLRequest(String ejbName, String methodName,
         Principal principal, Set<Principal> roles) throws Exception
   { 
      if(principal == null)
         throw new IllegalArgumentException("principal is null");
      
      String action = methodName; 
      
      RequestCtx requestCtx = null;  
      String username = principal.getName();
      
      //Create the subject set
      URI subjectAttrUri = new URI(XACMLConstants.SUBJECT_IDENTIFIER);
      Attribute subjectAttr = new Attribute(subjectAttrUri,null,null,
            new StringAttribute(username));
      Set<Attribute> subjectAttrSet = new HashSet<Attribute>();
      subjectAttrSet.add(subjectAttr);
      subjectAttrSet.addAll(getXACMLRoleSet(roles));
      
      Set<Subject> subjectSet = new HashSet<Subject>();
      subjectSet.add(new Subject(subjectAttrSet));
      
      //Create the resource set
      URI resourceUri = new URI(XACMLConstants.RESOURCE_IDENTIFIER);
      Attribute resourceAttr = new Attribute(resourceUri,null,null,
            new StringAttribute(ejbName));
      Set<Attribute> resourceSet = new HashSet<Attribute>();
      resourceSet.add(resourceAttr); 
      
      //Create the action set
      Set<Attribute> actionSet = new HashSet<Attribute>();
      actionSet.add(new Attribute(new URI(XACMLConstants.ACTION_IDENTIFIER),
             null,null, new StringAttribute(action)));
      
      
      //TODO: Get hold of the invocation arguments and populate in the xacml request
      
      //Create the Environment set
      Set<Attribute> environSet = new HashSet<Attribute>();
      //Current time
      URI currentTimeUri = new URI(XACMLConstants.CURRENT_TIME_IDENTIFIER);
      Attribute currentTimeAttr = new Attribute(currentTimeUri,null,null,
            new TimeAttribute());
      environSet.add(currentTimeAttr);
      
      //Create the request context
      requestCtx = new RequestCtx(subjectSet,resourceSet,actionSet,environSet);
      
      if(trace)
      {
         ByteArrayOutputStream baos = new ByteArrayOutputStream();
         requestCtx.encode(baos, new Indenter());
         log.trace("XACML Request:"+baos.toString());
         baos.close();
      }
      return requestCtx;
   } 
   
   public RequestCtx createXACMLRequest(String ejbName, String methodName,
         Principal principal, RoleGroup roles) throws Exception
   { 
      if(principal == null)
         throw new IllegalArgumentException("principal is null");
      if(roles == null)
         throw new IllegalArgumentException("roles is null");
      
      String action = methodName; 
      
      RequestCtx requestCtx = null;  
      String username = principal.getName();
      
      //Create the subject set
      URI subjectAttrUri = new URI(XACMLConstants.SUBJECT_IDENTIFIER);
      Attribute subjectAttr = new Attribute(subjectAttrUri,null,null,
            new StringAttribute(username));
      Set<Attribute> subjectAttrSet = new HashSet<Attribute>();
      subjectAttrSet.add(subjectAttr);
      subjectAttrSet.addAll(getXACMLRoleSet(roles));
      
      Set<Subject> subjectSet = new HashSet<Subject>();
      subjectSet.add(new Subject(subjectAttrSet));
      
      //Create the resource set
      URI resourceUri = new URI(XACMLConstants.RESOURCE_IDENTIFIER);
      Attribute resourceAttr = new Attribute(resourceUri,null,null,
            new StringAttribute(ejbName));
      Set<Attribute> resourceSet = new HashSet<Attribute>();
      resourceSet.add(resourceAttr); 
      
      //Create the action set
      Set<Attribute> actionSet = new HashSet<Attribute>();
      actionSet.add(new Attribute(new URI(XACMLConstants.ACTION_IDENTIFIER),
             null,null, new StringAttribute(action)));
      
      //TODO: Get hold of the invocation arguments and populate in the xacml request
      
      //Create the Environment set
      Set<Attribute> environSet = new HashSet<Attribute>();
      //Current time
      URI currentTimeUri = new URI(XACMLConstants.CURRENT_TIME_IDENTIFIER);
      Attribute currentTimeAttr = new Attribute(currentTimeUri,null,null,
            new TimeAttribute());
      environSet.add(currentTimeAttr);
      
      //Create the request context
      requestCtx = new RequestCtx(subjectSet,resourceSet,actionSet,environSet);
      
      if(trace)
      {
         ByteArrayOutputStream baos = new ByteArrayOutputStream();
         requestCtx.encode(baos, new Indenter());
         log.trace("XACML Request:"+baos.toString());
         baos.close();
      }
      return requestCtx;
   }
   
   private Set<Attribute> getXACMLRoleSet(Role role) throws Exception
   {
      
      Set<Attribute> roleset = new HashSet<Attribute>();
      
      if(role.getType() == RoleType.group)
      {
         RoleGroup rg = (RoleGroup) role;
         List<Role> roleList = rg.getRoles();
         for(Role r: roleList)
         {
           roleset.add(getRoleAttribute(r.getRoleName()));   
         }
      }
      else
         roleset.add(getRoleAttribute(role.getRoleName())); 
      return roleset;
   } 
   
   private Attribute getRoleAttribute(String roleName) throws URISyntaxException
   {
      URI roleURI = new URI(XACMLConstants.SUBJECT_ROLE_IDENTIFIER);
      return new Attribute(roleURI,null,null, new StringAttribute(roleName));
   }
   
   private Set<Attribute> getXACMLRoleSet(Set<Principal> roles) throws Exception
   {
      URI roleURI = new URI(XACMLConstants.SUBJECT_ROLE_IDENTIFIER);
   
      Set<Attribute> roleset = new HashSet<Attribute>();
      Iterator<Principal> iter = roles != null ? roles.iterator(): null;
      while(iter != null && iter.hasNext())
      {
         Principal role = iter.next();
         Attribute roleAttr = new Attribute(roleURI,null,null,
               new StringAttribute(role.getName()));
           roleset.add(roleAttr);  
      }
      return roleset;
   } 
}