/*
  * JBoss, Home of Professional Open Source.
  * Copyright 2006, Red Hat Middleware LLC, and individual contributors
  * as indicated by the @author tags. See the copyright.txt file in the
  * distribution for a full listing of individual contributors. 
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.mapping.providers;
 
import java.security.Principal;
import java.security.acl.Group;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.security.SecurityConstants;
import org.jboss.security.SimpleGroup;
import org.jboss.security.SimplePrincipal;
import org.jboss.security.mapping.MappingProvider;
import org.jboss.security.mapping.MappingResult;

//$Id$

/**
 *  A Role Mapping Module that takes into consideration a principal
 *  to roles mapping that can be done in the assembly descriptor of
 *  jboss.xml, jboss-web.xml and jboss-app.xml
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Nov 1, 2006 
 *  @version $Revision$
 */
public class DeploymentRolesMappingProvider implements MappingProvider<Group>
{
   private static Logger log = Logger.getLogger(DeploymentRolesMappingProvider.class);
   private boolean trace = log.isTraceEnabled();
   
   private MappingResult<Group> result;

   public void init(Map options)
   { 
   } 

   public void setMappingResult(MappingResult res)
   { 
      result = res;
   }

   /**
    * Obtains the deployment roles via the context map and applies it
    * on the mappedObject
    * @see MappingProvider#performMapping(Map, Object)
    */ 
   public void performMapping(Map map, Group mappedObject)
   {  
      if(map == null || map.isEmpty())
         throw new IllegalArgumentException("Context Map is null or empty");
    
      //Obtain the principal to roles mapping
      Principal principal = (Principal) map.get(SecurityConstants.PRINCIPAL_IDENTIFIER);
      Map<String,Set<String>> principalRolesMap = (Map<String,Set<String>>)map.get(SecurityConstants.DEPLOYMENT_PRINCIPAL_ROLES_MAP);
      if(trace)
         log.trace("Principal="+principal+":principalRolesMap="+principalRolesMap);
      
      if(principal == null || principalRolesMap == null || principalRolesMap.isEmpty())
         return ; // No Mapping
      
      Set<String> roleset = (Set<String>)principalRolesMap.get(principal.getName());
      if(roleset != null)
      {
         Group newRoles = new SimpleGroup(SecurityConstants.ROLES_IDENTIFIER);
         Iterator iter = roleset.iterator();
         while(iter.hasNext())
         {
            String rolename = (String)iter.next();
            newRoles.addMember(createNewPrincipal(mappedObject,rolename));
         }
         mappedObject = MappingProviderUtil.replacePrincipals(mappedObject, newRoles);  
      } 
      result.setMappedObject(mappedObject);
   } 
   
   /**
    * Need to maintain the Principal type from the original group
    * @param mappedObject
    * @param name
    * @return
    */
   private Principal createNewPrincipal(Group mappedObject, String name)
   {
      Principal p = new SimplePrincipal(name);
      
      //If the original group had a different principal than simpleprincipal
      if(mappedObject.members().hasMoreElements())
      {
         Principal origp = mappedObject.members().nextElement();
         p = MappingProviderUtil.instantiatePrincipal(origp.getClass(), name);
         if(p == null) 
            p = new SimplePrincipal(name); 
      }
      return p;
   } 
}