/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.acl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.jboss.security.authorization.Resource;
import org.jboss.security.identity.Identity;

/**
 * <p>
 * Simple ACL implementation that keeps the entries in a Map whose keys are the
 * identities of the entries, to provide fast access.
 * </p>
 * 
 * @author <a href="mailto:sguilhen@redhat.com">Stefan Guilhen</a>
 */
public class ACLImpl implements ACL
{

   private final Resource resource;

   private final Map<Identity, ACLEntry> entries;

   /**
    * <p>
    * Builds an instance of <code>ACLImpl</code> for the specified resource.
    * </p>
    * 
    * @param resource   a reference to the <code>Resource</code> associated with
    * the ACL being constructed.
    */
   public ACLImpl(Resource resource)
   {
      this(resource, new ArrayList<ACLEntry>());
   }

   /**
    * <p>
    * Builds an instance of <code>ACLImpl</code> for the specified resource, and initialize
    * it with the specified entries.
    * </p>
    * 
    * @param resource   a reference to the <code>Resource</code> associated with
    * the ACL being constructed.
    * @param entries    a <code>Collection</code> containing the ACL's initial entries.
    */
   public ACLImpl(Resource resource, Collection<ACLEntry> entries)
   {
      this.resource = resource;
      this.entries = new HashMap<Identity, ACLEntry>();
      for (ACLEntry entry : entries)
         this.entries.put(entry.getIdentity(), entry);
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACL#addEntry(org.jboss.security.acl.ACLEntry)
    */
   public boolean addEntry(ACLEntry entry)
   {
      // don't add a null entry or an entry that already exists.
      if (entry == null || this.entries.get(entry.getIdentity()) != null)
         return false;
      this.entries.put(entry.getIdentity(), entry);
      return true;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACL#removeEntry(org.jboss.security.acl.ACLEntry)
    */
   public boolean removeEntry(ACLEntry entry)
   {
      return (this.entries.remove(entry.getIdentity()) != null);
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACL#getEntries()
    */
   public Collection<ACLEntry> getEntries()
   {
      return Collections.unmodifiableCollection(this.entries.values());
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACL#getResource()
    */
   public Resource getResource()
   {
      return this.resource;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.security.acl.ACL#isGranted(org.jboss.security.acl.ACLPermission, org.jboss.security.identity.Identity)
    */
   public boolean isGranted(ACLPermission permission, Identity identity)
   {
      // lookup the entry corresponding to the specified identity.
      ACLEntry entry = this.entries.get(identity);
      if (entry != null)
      {
         // check the permission associated with the identity.
         return entry.checkPermission(permission);
      }
      return false;
   }

}
