/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.security;

import java.security.AccessController;
import java.security.Principal;
import java.security.PrivilegedAction;

import javax.security.auth.Subject;

import org.jboss.security.SecurityContextAssociation;

/** A PrivilegedAction implementation for setting the SecurityAssociation
 * principal and credential
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revison:$
 */
class SecurityAssociationActions
{
   private static class SetPrincipalInfoAction implements PrivilegedAction
   {
      Principal principal;
      Object credential;
      Subject subject;
      SetPrincipalInfoAction(Principal principal, Object credential, Subject subject)
      {
         this.principal = principal;
         this.credential = credential;
         this.subject = subject;
      }
      public Object run()
      {
         //SecurityAssociation.pushSubjectContext(subject, principal, credential);
         SecurityContext sc = SecurityContextAssociation.getSecurityContext();
         if(sc == null)
         {
            try
            {
               sc = SecurityContextFactory.createSecurityContext(principal, 
                                                credential, subject, "CLIENT_PROXY");
            }
            catch (Exception e)
            {
               throw new RuntimeException(e);
            }
         }
         SecurityContextAssociation.setSecurityContext(sc);
         credential = null;
         principal = null;
         subject = null;
         return null;
      }
   }
   private static class PopPrincipalInfoAction implements PrivilegedAction
   {
      public Object run()
      {
         //SecurityAssociation.popSubjectContext();
         SecurityContextAssociation.clearSecurityContext();
         return null;
      }
   }
   private static class SetServerAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new SetServerAction();
      public Object run()
      {
         SecurityAssociation.setServer();
         return null;
      }
   }
   private static class ClearAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new ClearAction();
      public Object run()
      {
         SecurityAssociation.clear();
         SecurityContextAssociation.clearSecurityContext();
         return null;
      }
   }
   private static class GetSubjectAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new GetSubjectAction();
      public Object run()
      {
         Subject subject = SecurityAssociation.getSubject();
         return subject;
      }
   }
   private static class GetPrincipalAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new GetPrincipalAction();
      public Object run()
      {
         Principal principal = SecurityAssociation.getPrincipal();
         return principal;
      }
   }
   private static class GetCredentialAction implements PrivilegedAction
   {
      static PrivilegedAction ACTION = new GetCredentialAction();
      public Object run()
      {
         Object credential = SecurityAssociation.getCredential();
         return credential;
      }
   }
   
   static void setSecurityContext(final SecurityContext sc)
   {
      AccessController.doPrivileged(new PrivilegedAction()
      { 
         public Object run()
         {
            SecurityContextAssociation.setSecurityContext(sc); 
            return null;
         }
      });
   }
   
   static SecurityContext getSecurityContext()
   {
      return (SecurityContext)AccessController.doPrivileged(new PrivilegedAction()
      { 
         public Object run()
         {
            return SecurityContextAssociation.getSecurityContext(); 
         }
      });
   }
   
   static void pushSecurityContext(final Principal p, final Object cred, 
         final Subject subject, final String securityDomain)
   {
      AccessController.doPrivileged(new PrivilegedAction()
      { 
         public Object run()
         {
            SecurityContext sc;
            try
            {
               sc = SecurityContextFactory.createSecurityContext(p, cred, 
                     subject, securityDomain);
            }
            catch (Exception e)
            {
               throw new RuntimeException(e);
            }
            SecurityContextAssociation.setSecurityContext(sc);
            return null;
         }
      });
   }

   static void setPrincipalInfo(Principal principal, Object credential, Subject subject)
   {
      SetPrincipalInfoAction action = new SetPrincipalInfoAction(principal, credential, subject);
      AccessController.doPrivileged(action);
   }
   static void popPrincipalInfo()
   {
      PopPrincipalInfoAction action = new PopPrincipalInfoAction();
      AccessController.doPrivileged(action);
   }
   static void setServer()
   {
      AccessController.doPrivileged(SetServerAction.ACTION);
   }
   static void clear()
   {
      AccessController.doPrivileged(ClearAction.ACTION);
   }
   static Subject getSubject()
   {
      Subject subject = (Subject) AccessController.doPrivileged(GetSubjectAction.ACTION);
      return subject;
   }
   static Principal getPrincipal()
   {
      Principal principal = (Principal) AccessController.doPrivileged(GetPrincipalAction.ACTION);
      return principal;
   }
   static Object getCredential()
   {
      Object credential = AccessController.doPrivileged(GetCredentialAction.ACTION);
      return credential;
   }

}
