/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.authorization.sunxacml;

import java.io.ByteArrayOutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.security.authorization.AuthorizationContext;
import org.jboss.security.authorization.XACMLConstants;

import com.sun.xacml.Indenter;
import com.sun.xacml.PDP;
import com.sun.xacml.PDPConfig;
import com.sun.xacml.Policy;
import com.sun.xacml.ctx.RequestCtx;
import com.sun.xacml.ctx.ResponseCtx;
import com.sun.xacml.ctx.Result;
import com.sun.xacml.finder.AttributeFinder;
import com.sun.xacml.finder.PolicyFinder;
import com.sun.xacml.finder.impl.CurrentEnvModule;
import com.sun.xacml.finder.impl.SelectorModule;
import com.sun.xacml.support.finder.URLPolicyFinderModule;

//$Id: JBossXACMLUtil.java 68749 2008-01-09 20:25:39Z anil.saldhana@jboss.com $

/**
 *  Util class for XACML Integration
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jun 21, 2006 
 *  @version $Revision: 68749 $
 */
public class JBossXACMLUtil
{
   private static Logger log = Logger.getLogger(JBossXACMLUtil.class);
   private static boolean trace = log.isTraceEnabled(); 
   
   /**
    * Peek into the XACML Reponse to derive the result 
    * of authorization (either AuthorizationContext.PERMIT or
    * AuthorizationContext.DENY)
    * Also, @see JBossXACMLUtil#checkXACMLAuthorization(RequestCtx, Policy)
    * @param response xacml response
    * @return AuthorizationContext.PERMIT or AuthorizationContext.DENY
    * @throws Exception
    */
   public static int analyzeResponseCtx(ResponseCtx response) throws Exception
   {
      if(trace)
      {
         ByteArrayOutputStream baos = new ByteArrayOutputStream();
         response.encode(baos, new Indenter());
         log.trace("XACML Response:"+baos.toString());
         baos.close();
      }
      int result = AuthorizationContext.DENY;
      Set<Result> results = response.getResults();
      if(results.size() > 1)
         throw new IllegalArgumentException("Number of results > 1");
      Iterator<Result> iter = results.iterator(); 
      if(iter.hasNext())
      {
         Result res = (Result)iter.next();
         int decision = res.getDecision();
         if(decision == Result.DECISION_PERMIT)
            result = AuthorizationContext.PERMIT; 
      }
      return result;
   }
   

   /**
    * Perform XACML Authorization in one step
    * @param request xacml request
    * @param policy xacml Policy Object
    * @return AuthorizationContext.PERMIT or AuthorizationContext.DENY
    * @throws Exception
    */
   public static synchronized int checkXACMLAuthorization(RequestCtx request,
         Policy policy) throws Exception
   {
      ResponseCtx response = getXACMLResponse(request,policy);
      return analyzeResponseCtx(response);
   } 
   
   /**
    * Get the default policy id created for dynamic policy files
    * @return
    */
   public static URI getDefaultPolicyID()
   {
      URI policyID = null;
      try
      {
         policyID = new URI(XACMLConstants.JBOSS_DYNAMIC_POLICY_SET_IDENTIFIER);
      }
      catch(Exception e)
      {
         log.debug("Exception in getDefaultPolicyID:",e);
      }
      return policyID;
   }
   
   /**
    * Invoke the XACML PDP
    * Also, @see JBossXACMLUtil#checkXACMLAuthorization(RequestCtx, Policy)
    * @param request xacml request
    * @param policy XACML Policy object
    * @return xacml response
    */
   public static synchronized ResponseCtx getXACMLResponse(RequestCtx request,
         Policy policy)
   {
      AttributeFinder attributeFinder = new AttributeFinder();
      List attributeModules = new ArrayList();  
      attributeModules.add(new CurrentEnvModule());
      attributeModules.add(new SelectorModule());
      attributeFinder.setModules(attributeModules); 
      
      //List policyFileList = Arrays.asList(policyFiles);
      PolicyFinder policyFinder = new PolicyFinder();
      HashSet policyModules = new HashSet();
      policyModules.add(new EnclosingPolicyFinderModule(policy));
      /*policyModules.add(new JBossStaticPolicyFinderModule(PermitOverridesPolicyAlg.algId,
            policyFileList));*/
      //policyModules.add(new StaticRefPolicyFinderModule(policyFileList));
      policyModules.add(new URLPolicyFinderModule());
      policyFinder.setModules(policyModules); 
      
      PDP pdp = new PDP(new PDPConfig(attributeFinder, 
            policyFinder, null));
      return pdp.evaluate(request);
   } 
   
}
