/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.config;

import java.security.Principal;

import org.jboss.security.auth.login.BaseAuthenticationInfo;
import org.jboss.security.identity.RoleGroup;

// $Id: ApplicationPolicy.java 75413 2008-07-06 14:47:34Z sguilhen@redhat.com $

/**
 * Application Policy Information Holder - Authentication - Authorization - Audit - Mapping
 * 
 * @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 * @since Jun 9, 2006
 * @version $Revision: 75413 $
 */
public class ApplicationPolicy
{
   private final String name;

   private BaseAuthenticationInfo authenticationInfo;

   private ACLInfo aclInfo;

   private AuthorizationInfo authorizationInfo;

   private AuditInfo auditInfo;

   private MappingInfo roleMappingInfo;

   private MappingInfo principalMappingInfo;

   private IdentityTrustInfo identityTrustInfo;

   // Base application policy (if any)
   private String baseApplicationPolicyName;

   // Parent PolicyConfig
   private PolicyConfig policyConfig;

   public ApplicationPolicy(String theName)
   {
      if (theName == null)
         throw new IllegalArgumentException("name is null");
      this.name = theName;
   }

   public ApplicationPolicy(String theName, BaseAuthenticationInfo info)
   {
      this(theName);
      authenticationInfo = info;
   }

   public ApplicationPolicy(String theName, AuthorizationInfo info)
   {
      this(theName);
      authorizationInfo = info;
   }

   public ApplicationPolicy(String theName, BaseAuthenticationInfo info, AuthorizationInfo info2)
   {
      this(theName);
      authenticationInfo = info;
      authorizationInfo = info2;
   }

   public ACLInfo getAclInfo()
   {
      ACLInfo info = null;
      ApplicationPolicy basePolicy = this.getBaseApplicationPolicy();
      if (basePolicy != null)
         info = basePolicy.getAclInfo();
      if (info != null && this.aclInfo == null)
         return info;
      else if (info != null)
         return (ACLInfo) this.aclInfo.merge(info);
      else
         return aclInfo;
   }

   public void setAclInfo(ACLInfo aclInfo)
   {
      this.aclInfo = aclInfo;
   }

   public BaseAuthenticationInfo getAuthenticationInfo()
   {
      BaseAuthenticationInfo bai = null;
      ApplicationPolicy ap = this.getBaseApplicationPolicy();
      if (ap != null)
         bai = ap.getAuthenticationInfo();
      if (bai != null && authenticationInfo == null)
         return bai;
      else if (bai != null)
         return (BaseAuthenticationInfo) authenticationInfo.merge(bai);
      else
         return authenticationInfo;
   }

   public void setAuthenticationInfo(BaseAuthenticationInfo authenticationInfo)
   {
      this.authenticationInfo = authenticationInfo;
   }

   public AuthorizationInfo getAuthorizationInfo()
   {
      AuthorizationInfo bai = null;
      ApplicationPolicy ap = this.getBaseApplicationPolicy();
      if (ap != null)
         bai = ap.getAuthorizationInfo();
      if (bai != null && authorizationInfo == null)
         return bai;
      else if (bai != null)
         return (AuthorizationInfo) authorizationInfo.merge(bai);
      else
         return authorizationInfo;
   }

   public void setAuthorizationInfo(AuthorizationInfo authorizationInfo)
   {
      this.authorizationInfo = authorizationInfo;
   }

   public MappingInfo getRoleMappingInfo()
   {
      MappingInfo bai = null;
      ApplicationPolicy ap = this.getBaseApplicationPolicy();
      if (ap != null)
         bai = ap.getRoleMappingInfo();

      if (bai != null && roleMappingInfo == null)
         return bai;
      else if (bai != null)
         return (MappingInfo) roleMappingInfo.merge(bai);
      else
         return roleMappingInfo;
   }

   public void setRoleMappingInfo(MappingInfo roleMappingInfo)
   {
      this.roleMappingInfo = roleMappingInfo;
   }

   public MappingInfo getPrincipalMappingInfo()
   {
      MappingInfo bai = null;
      ApplicationPolicy ap = this.getBaseApplicationPolicy();
      if (ap != null)
         bai = ap.getPrincipalMappingInfo();

      if (bai != null && principalMappingInfo == null)
         return bai;
      else if (bai != null)
         return (MappingInfo) principalMappingInfo.merge(bai);
      else
         return principalMappingInfo;

   }

   public void setPrincipalMappingInfo(MappingInfo principalMappingInfo)
   {
      this.principalMappingInfo = principalMappingInfo;
   }

   public <T> MappingInfo getMappingInfo(Class<T> t)
   {
      if (t == RoleGroup.class)
         return this.getRoleMappingInfo();
      if (t == Principal.class)
         return this.getPrincipalMappingInfo();
      throw new IllegalStateException("No mapping information available for type:" + t);
   }

   public AuditInfo getAuditInfo()
   {
      AuditInfo bai = null;
      ApplicationPolicy ap = this.getBaseApplicationPolicy();
      if (ap != null)
         bai = ap.getAuditInfo();

      if (bai != null && auditInfo == null)
         return bai;
      else if (bai != null)
         return (AuditInfo) auditInfo.merge(bai);
      else
         return auditInfo;
   }

   public void setAuditInfo(AuditInfo auditInfo)
   {
      this.auditInfo = auditInfo;
   }

   public IdentityTrustInfo getIdentityTrustInfo()
   {
      IdentityTrustInfo bai = null;
      ApplicationPolicy ap = this.getBaseApplicationPolicy();
      if (ap != null)
         bai = ap.getIdentityTrustInfo();

      if (bai != null && identityTrustInfo == null)
         return bai;
      else if (bai != null)
         return (IdentityTrustInfo) identityTrustInfo.merge(bai);
      else
         return identityTrustInfo;
   }

   public void setIdentityTrustInfo(IdentityTrustInfo identityTrustInfo)
   {
      this.identityTrustInfo = identityTrustInfo;
   }

   public String getBaseApplicationPolicyName()
   {
      return baseApplicationPolicyName;
   }

   public void setBaseApplicationPolicyName(String baseApplicationPolicy)
   {
      this.baseApplicationPolicyName = baseApplicationPolicy;
   }

   public String getName()
   {
      return name;
   }

   public PolicyConfig getPolicyConfig()
   {
      return policyConfig;
   }

   public void setPolicyConfig(PolicyConfig policyConfig)
   {
      this.policyConfig = policyConfig;
   }

   private ApplicationPolicy getBaseApplicationPolicy()
   {
      ApplicationPolicy ap = null;
      if (this.baseApplicationPolicyName != null)
      {
         ap = this.policyConfig.get(this.baseApplicationPolicyName);
         // The base application policy may exist in a different location
         if (ap == null)
            ap = SecurityConfiguration.getApplicationPolicy(this.baseApplicationPolicyName);
      }
      return ap;
   }
}