/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.authorization.modules.web;

import java.io.ByteArrayOutputStream;
import java.net.URI;
import java.security.Principal;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.jboss.logging.Logger;
import org.jboss.security.AuthorizationManager;
import org.jboss.security.SimplePrincipal;
import org.jboss.security.authorization.XACMLConstants;

import com.sun.xacml.Indenter;
import com.sun.xacml.attr.AnyURIAttribute;
import com.sun.xacml.attr.StringAttribute;
import com.sun.xacml.attr.TimeAttribute;
import com.sun.xacml.ctx.Attribute;
import com.sun.xacml.ctx.RequestCtx;
import com.sun.xacml.ctx.Subject;

//$Id: WebXACMLUtil.java 46543 2006-07-27 20:22:05Z asaldhana $

/**
 *  Utility class for creating XACML Requests
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jun 21, 2006 
 *  @version $Revision: 46543 $
 */
public class WebXACMLUtil
{
   private static Logger log = Logger.getLogger(WebXACMLUtil.class);
   private boolean trace = log.isTraceEnabled();
   
   public WebXACMLUtil()
   {   
   }
   
   public RequestCtx createXACMLRequest(HttpServletRequest request,
         AuthorizationManager authzManager, 
         Set<Principal> roles) throws Exception
   { 
      if(request == null)
         throw new IllegalArgumentException("Http Request is null");
      if(authzManager == null)
         throw new IllegalArgumentException("Authorization Manager is null");
      String httpMethod = request.getMethod();
      String action = "GET".equals(httpMethod)?"read":"write";
      
      //Non-standard uri
      String actionURIBase = XACMLConstants.JBOSS_RESOURCE_PARAM_IDENTIFIER;
      
      RequestCtx requestCtx = null;
      Principal principal = request.getUserPrincipal();
      String username = principal.getName();  
      //Create the subject set
      URI subjectAttrUri = new URI(XACMLConstants.SUBJECT_IDENTIFIER);
      Attribute subjectAttr = new Attribute(subjectAttrUri,null,null,
            new StringAttribute(username));
      Set subjectAttrSet = new HashSet();
      subjectAttrSet.add(subjectAttr);
      subjectAttrSet.addAll(getXACMLRoleSet(roles));
      
      Set subjectSet = new HashSet();
      subjectSet.add(new Subject(subjectAttrSet));
      
      //Create the resource set
      URI resourceUri = new URI(XACMLConstants.RESOURCE_IDENTIFIER);
      Attribute resourceAttr = new Attribute(resourceUri,null,null,
            new AnyURIAttribute(new URI(request.getRequestURI())));
      Set resourceSet = new HashSet();
      resourceSet.add(resourceAttr); 
      
      //Create the action set
      Set actionSet = new HashSet();
      actionSet.add(new Attribute(new URI(XACMLConstants.ACTION_IDENTIFIER),
             null,null, new StringAttribute(action)));
      
      Enumeration<String> enumer = request.getParameterNames();
      while(enumer.hasMoreElements())
      {
         String paramName = enumer.nextElement();
         String paramValue = request.getParameter(paramName);
         URI actionUri = new URI(actionURIBase + paramName);
         Attribute actionAttr = new Attribute(actionUri,null,null,
               new StringAttribute(paramValue));
         actionSet.add(actionAttr); 
      }
      //Create the Environment set
      Set environSet = new HashSet();
      //Current time
      URI currentTimeUri = new URI(XACMLConstants.CURRENT_TIME_IDENTIFIER);
      Attribute currentTimeAttr = new Attribute(currentTimeUri,null,null,
            new TimeAttribute());
      environSet.add(currentTimeAttr);
      
      //Create the request context
      requestCtx = new RequestCtx(subjectSet,resourceSet,actionSet,environSet);
      
      if(trace)
      {
         ByteArrayOutputStream baos = new ByteArrayOutputStream();
         requestCtx.encode(baos, new Indenter());
         log.trace("XACML Request:"+baos.toString());
         baos.close();
      }
      return requestCtx;
   } 
   
   private Set<Attribute> getXACMLRoleSet(Set<Principal> roles) throws Exception
   {
      URI roleURI = new URI(XACMLConstants.SUBJECT_ROLE_IDENTIFIER);
   
      Set<Attribute> roleset = new HashSet<Attribute>();
      Iterator<Principal> iter = roles != null ? roles.iterator(): null;
      while(iter != null && iter.hasNext())
      {
         Principal role = iter.next();
         if(role instanceof SimplePrincipal)
         {
            SimplePrincipal sp = (SimplePrincipal)role;
            Attribute roleAttr = new Attribute(roleURI,null,null,
                new StringAttribute(sp.getName()));
            roleset.add(roleAttr); 
         }
      }
      return roleset;
   } 
}