/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */ 
package org.jboss.security.mapping.providers;
  
import java.security.Principal;
import java.security.acl.Group;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Map;
import java.util.Properties;

import org.jboss.logging.Logger;
import org.jboss.security.mapping.MappingProvider;
import org.jboss.security.mapping.MappingResult;


/**
 *  Role Mapping Provider that picks up the roles from the
 *  options and then appends them to the passed Group
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @version $Revision$
 *  @since  Aug 24, 2006
 */
public class OptionsRoleMappingProvider implements MappingProvider<Group> 
{
   private static final Logger log = Logger.getLogger(OptionsRoleMappingProvider.class);
   
   //Standard Strings
   private static final String REPLACE_ROLES_STRING = "replaceRoles";
   private static final String ROLES_MAP = "rolesMap";
   
   private MappingResult<Group> result;

   private Map options = null;
   
   private Properties roleMapProperties = null;
   
   /**
    * Specifies
    */
   private boolean REPLACE_ROLES = false;
   
   public void init(Map opt)
   {
     this.options = opt;
     if(options != null)
     {
        if(options.containsKey(REPLACE_ROLES_STRING))
        {
           REPLACE_ROLES = "true".equalsIgnoreCase((String)options.get(REPLACE_ROLES_STRING)); 
        }
        if(options.containsKey(ROLES_MAP))
        {
           roleMapProperties = (Properties)options.get(ROLES_MAP);
        } 
     } 
   }
   
   public void setMappingResult(MappingResult res)
   { 
      result = res;
   }
   
   public void performMapping(Map contextMap, Group mappedObject)
   { 
      ArrayList<Principal> removeMembers = new ArrayList<Principal>();
      ArrayList<Principal> addMembers = new ArrayList<Principal>(); 

      //Enumerate over the members
      Enumeration enumer = mappedObject.members();
      while(enumer.hasMoreElements())
      {
         Principal p = (Principal)enumer.nextElement();
         String name = p.getName(); 
         String commaSeparatedRoles = roleMapProperties.getProperty(name);
         if(commaSeparatedRoles != null)
         {
            String[] tokens = MappingProviderUtil.getRolesFromCommaSeparatedString(commaSeparatedRoles);
            int len = tokens != null ? tokens.length : 0;
            for(int i = 0; i < len; i++)
            {
               if(this.REPLACE_ROLES)
                  removeMembers.add(p); 
               addMembers.add(MappingProviderUtil.instantiatePrincipal(p.getClass(),tokens[i])); 
            }
         } 
      }
      //Go through  the remove list
      for(Principal p:removeMembers)
      {
         mappedObject.removeMember(p);
      }
      //Go through the add list
      for(Principal p:addMembers)
      {
         mappedObject.addMember(p);
      }
      
      result.setMappedObject(mappedObject);
   }   
}