/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.security.plugins;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.CodeSource;
import java.security.PermissionCollection;
import java.security.Policy;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.security.ProtectionDomain;
import java.security.cert.Certificate;
import java.util.LinkedList;

import org.jboss.logging.Logger;
import org.jboss.security.PushPermission;
import org.jboss.virtual.VFSUtils;
import org.jboss.virtual.VirtualFile;

/**
 * Class that can be used to build own stack of protection domains and do
 * privileged operations against it. Example would be getting property value
 * injected to the deployed code even if the code deployed doesn't have
 * permission to read it.
 * 
 * @author baranowb
 * @author pskopek
 * 
 */
public class AccessControlContextManipulator {

   private static final Logger log = Logger
         .getLogger(AccessControlContextManipulator.class);

   private static final AccessControlContext CONTEXT_EMPTY = new AccessControlContext(
         new ProtectionDomain[0]);
   private static final ThreadLocal<LinkedList<ProtectionDomain>> THREAD_DOMAINS = new ThreadLocal<LinkedList<ProtectionDomain>>();
   private static final ThreadLocal<AccessControlContext> THREAD_CONTEXT = new ThreadLocal<AccessControlContext>();

   /**
    * Push virtual file to context for later permission  check.
    * 
    * @param file
    * @throws MalformedURLException
    * @throws IOException
    * @throws URISyntaxException
    */
   public static void pushContext(final VirtualFile file)
         throws MalformedURLException, IOException, URISyntaxException {

      SecurityManager sm = System.getSecurityManager();  
            
      if (sm == null) {
         return;
      }

      // check if we have pushPermission
      sm.checkPermission(new PushPermission());
      
      VirtualFile pf = file.getParent();

      VirtualFile root = null;
      if (pf.getPathName().endsWith("META-INF")
            || pf.getPathName().endsWith("META-INF/")) {
         root = pf.getParent();
      } else {
         root = file;
      }

      URL realRootURL = VFSUtils.getRealURL(root);

      AccessControlContextManipulator.pushContext(realRootURL.toURI());

   }

   /**
    * Push uri to context for later permission  check.
    * 
    * @param uri
    * @throws IOException
    */
   public static void pushContext(final URI uri) throws IOException {

      SecurityManager sm = System.getSecurityManager();  
      
      // policyfile should handle empty certs when it processes CS
      if (sm == null || Policy.getPolicy() == null) {
         return;
      }
      
      // check if we have pushPermission
      sm.checkPermission(new PushPermission());

      if (log.isTraceEnabled()) {
         log.trace("Pushing " + uri + " to context.");
      }

      initThreadLocal();

      ProtectionDomain protectionDomain = null;
      final CodeSource codeSource = new CodeSource(uri.toURL(),
            (Certificate[]) null);
      final PermissionCollection permissionCollection = Policy.getPolicy()
            .getPermissions(codeSource);
      protectionDomain = new ProtectionDomain(codeSource, permissionCollection);

      THREAD_DOMAINS.get().addLast(protectionDomain);
      THREAD_CONTEXT.set(null);

   }

   /**
    * Remove last protection domain from context. 
    */
   public static void popContext() {
      if (System.getSecurityManager() == null || Policy.getPolicy() == null) {
         return;
      }
      initThreadLocal();
      log.trace("Poping from context");
      
      LinkedList<ProtectionDomain> list = THREAD_DOMAINS.get();
      if (list.isEmpty()) {
         return;
      }
      THREAD_DOMAINS.get().removeLast();
      THREAD_CONTEXT.set(null);
   }

   public static <T> T doPrivileged(PrivilegedAction<T> action) {
      return AccessController.doPrivileged(action, getCurrentContext());
   }

   public static <T> T doPrivileged(PrivilegedExceptionAction<T> action)
         throws PrivilegedActionException {
      return AccessController.doPrivileged(action, getCurrentContext());
   }

   private static AccessControlContext getCurrentContext() {

      boolean trace = log.isTraceEnabled();

      AccessControlContext context = THREAD_CONTEXT.get();
      if (context != null) {
         if (trace)
            log.trace("returning context directly " + context);
         return context;
      }
      LinkedList<ProtectionDomain> protectionDomains = THREAD_DOMAINS.get();
      if (protectionDomains == null || protectionDomains.isEmpty()) {
         if (trace)
            log.trace("returning empty context");
         return CONTEXT_EMPTY;
      }

      if (trace) {
         int i = 0;
         for (ProtectionDomain p : protectionDomains) {
            log.trace(i++ + ". PD = " + p);
         }
      }

      ProtectionDomain[] protectionDomainsArray = new ProtectionDomain[protectionDomains
            .size()];
      protectionDomainsArray = protectionDomains
            .toArray(protectionDomainsArray);
      context = new AccessControlContext(protectionDomainsArray);

      THREAD_CONTEXT.set(context);
      return context;
   }

   private static void initThreadLocal() {
      if (THREAD_DOMAINS.get() == null) {
         THREAD_DOMAINS.set(new LinkedList<ProtectionDomain>());
      }
   }

}
