/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.xacml.existdb.locators.policy;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import org.jboss.security.xacml.bridge.PolicySetFinderModule;
import org.jboss.security.xacml.existdb.ExistDBConstants;
import org.jboss.security.xacml.existdb.ExistSecurityCtx;
import org.jboss.security.xacml.existdb.util.ExistDBUtil;
import org.jboss.security.xacml.factories.PolicyFactory;
import org.jboss.security.xacml.interfaces.XACMLConstants;
import org.jboss.security.xacml.interfaces.XACMLPolicy;
import org.jboss.security.xacml.jaxb.Option;
import org.jboss.security.xacml.locators.JBossPolicySetLocator; 
import org.xmldb.api.DatabaseManager;
import org.xmldb.api.base.Collection;
import org.xmldb.api.base.Database;

/**
 * A Policy Set Locator that gets the policies from the ExistDB
 * @author Anil.Saldhana@redhat.com
 * @since Mar 24, 2010
 */
@SuppressWarnings("unchecked")
public class ExistPolicySetLocator extends JBossPolicySetLocator
{
   static
   {
      try
      {
         // initialize driver
         String driver = "org.exist.xmldb.DatabaseImpl";
         Class cl = Class.forName(driver);
         Database database = (Database)cl.newInstance();
         database.setProperty("create-database", "true");
         DatabaseManager.registerDatabase(database); 
      } 
      catch(Exception e)
      {
         throw new RuntimeException("Error initializing the ExistDB Drivers:",e);
      }
   }
   
   public ExistPolicySetLocator()
   {
      super(); 
      PolicySetFinderModule finderModule = new PolicySetFinderModule();
      List<PolicySetFinderModule> finders = new ArrayList<PolicySetFinderModule>();
      finders.add(finderModule);
      
      this.set(XACMLConstants.POLICY_FINDER_MODULE, finders);
   }

   public ExistPolicySetLocator(Set<XACMLPolicy> policies)
   {
      super(policies); 
   }

   
   @Override
   public void setOptions(List<Option> theoptions)
   { 
      super.setOptions(theoptions); 
      Set<XACMLPolicy> policies = getPolicies();
      this.setPolicies(policies);
   }

   @Override
   public Set<XACMLPolicy> getPolicies()
   {
      Set<XACMLPolicy> policies = new HashSet<XACMLPolicy>();
      
      String uri = ExistDBUtil.getOptionValue(options,ExistDBConstants.DATABASE_URL);
      if(uri == null)
         throw new RuntimeException("database url not configured");
      
      String collectionName = ExistDBUtil.getOptionValue(options,ExistDBConstants.COLLECTION_NAME_KEY);
      
      if(collectionName == null)
         collectionName = "xacml";   
      
      String documentName = ExistDBUtil.getOptionValue(options,ExistDBConstants.DOCUMENT_NAME_KEY);
      String namesOfDocuments = ExistDBUtil.getOptionValue(options,ExistDBConstants.NAMES_OF_DOCUMENTS_KEY);
      
      if(documentName == null && namesOfDocuments == null)
         throw new RuntimeException("documentName or namesofDocuments not configured in the ExistPolicySetLocator");
      
      String userName = ExistDBUtil.getOptionValue(options,ExistDBConstants.USERNAME_KEY);
      String pass = ExistDBUtil.getOptionValue(options,ExistDBConstants.PASSWORD_KEY); 
      
      try
      {
         Collection collection = null;
         
         if(userName == null || userName != "")
            collection = ExistDBUtil.getCollection(uri, collectionName, null);
         else
            collection = ExistDBUtil.getCollection(uri, collectionName, new ExistSecurityCtx(userName, pass));
         
         if(namesOfDocuments != null)
         {
            String[] docNames = getTokens(namesOfDocuments);
            for(String docName: docNames)
            {
               String policy = ExistDBUtil.retrieve(collection, docName);
               ByteArrayInputStream bis = new ByteArrayInputStream(policy.getBytes("UTF-8"));
               
               XACMLPolicy xacmlPolicySet = createPolicy(bis);
               policies.add(xacmlPolicySet); 
            } 
         }
         else
         {   
            String policy = ExistDBUtil.retrieve(collection, documentName);
            ByteArrayInputStream bis = new ByteArrayInputStream(policy.getBytes("UTF-8"));
            
            XACMLPolicy xacmlPolicySet = createPolicy(bis);
            policies.add(xacmlPolicySet); 
         }
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
 
      return policies;
   } 

   /**
    * Method that is used to create the correct xacml policy or policyset
    * @param is
    * @return
    * @throws Exception
    */
   protected XACMLPolicy createPolicy(InputStream is) throws Exception
   {
      return PolicyFactory.createPolicySet(is);
   }
   
   /**
    * Get the comma separated tokens
    * @param csv
    * @return
    */
   private String[] getTokens(String csv)
   {
      ArrayList<String> alist = new ArrayList<String>();
      if(csv == null)
         throw new IllegalArgumentException("csv is null");
      
      StringTokenizer st = new StringTokenizer(csv, ",");
      if(st != null)
      {
         while(st.hasMoreTokens())
         {
            alist.add(st.nextToken()); 
         }
      }
      String[] strArr = new String[alist.size()];
      alist.toArray(strArr);
      return strArr;
   }
}