/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.xacml.existdb.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.util.List;

import javax.xml.transform.OutputKeys;

import org.exist.storage.serializers.EXistOutputKeys;
import org.exist.xmldb.XQueryService;
import org.jboss.security.xacml.existdb.ExistSecurityCtx;
import org.jboss.security.xacml.jaxb.Option;
import org.xmldb.api.DatabaseManager;
import org.xmldb.api.base.Collection;
import org.xmldb.api.base.Resource;
import org.xmldb.api.base.ResourceIterator;
import org.xmldb.api.base.ResourceSet;
import org.xmldb.api.base.XMLDBException;
import org.xmldb.api.modules.XMLResource;

/**
 * @author Anil.Saldhana@redhat.com
 * @since Mar 24, 2010
 */
public class ExistDBUtil
{ 
   /**
    * Given the list of options, get the value
    * @param options
    * @param key the value to be returned has this key
    * @return
    */
   public static String getOptionValue(List<Option> options, String key)
   {
      for(Option option: options)
      {
         if(key.equals(option.getName()) )
           return (String) option.getContent().get(0);        
      } 
      return null;
   }
   
   /**
    * Given a file instance, try to read it into a String as the XML DB api
    * use a string for xquery
    * <b>NOTE:</b> If the file does not exist, an attempt is made to read
    * the file via the Thread Context Class Loader.
    * 
    * @param file
    * @return
    */
   public static String readStaticFileIntoString(File file)
   {
      Reader reader = null;
      if(file.exists() == false)
      {
         //Use other mechanisms to get to the file
         ClassLoader tcl = SecurityActions.getContextClassLoader();
         String path = file.getPath();
         
         InputStream is =  tcl.getResourceAsStream(file.getPath());
         if(is == null)
            throw new RuntimeException("file could not be located " +  path);
         
         try
         {
            reader = new InputStreamReader(is, "UTF-8");
         }
         catch (UnsupportedEncodingException e)
         {
            throw new RuntimeException(e);
         }
      }
      else
      {
         try
         {
            reader = new FileReader(file);
         }
         catch (FileNotFoundException e)
         {
            throw new RuntimeException(e);
         }
      }
      StringBuilder contents = new StringBuilder();
      
      try 
      {
        BufferedReader input =  new BufferedReader(reader);
        try 
        {
          String line = null;  
          while (( line = input.readLine()) != null)
          {
            contents.append(line);
            contents.append(SecurityActions.getSystemProperty("line.separator"));
          }
        }
        finally 
        {
          input.close();
        }
      }
      catch (IOException ex)
      {
         throw new RuntimeException(ex); 
      }
      
      return contents.toString(); 
   }
   
   /**
    * Execute an XQuery against the Exist DB
    * @param collection
    * @param query
    * @return
    * @throws Exception
    */
   public static Object query(Collection collection, String query) throws Exception
   { 
      XQueryService service = (XQueryService)
      collection.getService("XQueryService", "1.0");
      ResourceSet result = service.query(query);
      long size = result.getSize();
      if(size > 1)
         throw new RuntimeException("XQuery result is greater than 1");
      
      ResourceIterator resourceIterator = result.getIterator();
      Resource resource = resourceIterator.nextResource();
      return resource.getContent();  
   }
   
   /**
    * Retrieve the policy as a string
    * @param collection
    * @param documentName
    * @return
    * @throws Exception
    */
   public static String retrieve(Collection collection, String documentName) throws Exception
   { 
        XMLResource res = (XMLResource)collection.getResource(documentName);
        if(res == null)
           throw new RuntimeException("Document not found in exist db:" + documentName + " in collection:" + 
                 collection.getName());
        
        Object content = res.getContent();
        return (String) content; 
   }
   
   /**
    * Given the database URI and the collection name, get the Collection.
    * @param URI
    * @param collectionName
    * @param sctx
    * @return
    * @throws XMLDBException
    */
   public static Collection getCollection(String URI, String collectionName, ExistSecurityCtx sctx) throws XMLDBException
   {
      Collection col = null;
      if (sctx != null)
         col = DatabaseManager.getCollection(URI + collectionName, sctx.getUsername(), sctx.getPassword());
      else
         col = DatabaseManager.getCollection(URI + collectionName);
      col.setProperty(OutputKeys.INDENT, "no");
      col.setProperty(EXistOutputKeys.EXPAND_XINCLUDES, "no"); 
      col.setProperty(EXistOutputKeys.PROCESS_XSL_PI, "no"); 
      return col;  
   } 
}