/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.server.jboss;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.soa.dsp.server.ServerConfig;
import org.jboss.util.id.GUID;

import javax.management.AttributeNotFoundException;
import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import java.io.File;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Set;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class JBossServerConfigImpl implements JBossServerConfig
{
  protected final Log log = LogFactory.getLog(getClass());

  // The MBeanServer
  private MBeanServer mbeanServer;

  private final static String UNDEFINED_HOSTNAME = "undefined.host";

  // The webservice host name that will be used when updating the wsdl
  private String webServiceHost = UNDEFINED_HOSTNAME;
  // The webservice port that will be used when updating the wsdl
  private int webServicePort;
  // The webservice port that will be used when updating the wsdl
  private int webServiceSecurePort;
  
  private static String nonClusteredServerName=new GUID().toString();
  
  private String clusterNodeName;

  public JBossServerConfigImpl()
  {
  }

  public MBeanServer getMbeanServer()
  {
    return mbeanServer;
  }

  public void setMbeanServer(MBeanServer mbeanServer)
  {
    this.mbeanServer = mbeanServer;
  }

  public String getImplementationTitle()
  {
    return "jbossas-riftsaw";
  }

  public String getImplementationVersion()
  {
    return "5.x";
  }

  public File getServerTempDir()
  {
    try
    {
      ObjectName oname = ObjectNameFactory.create("jboss.system:type=ServerConfig");
      File dir = (File)getMbeanServer().getAttribute(oname, "ServerTempDir");
      return dir;
    }
    catch (JMException e)
    {
      return null;
    }
  }

  public File getHomeDir()
  {
    try
    {
      ObjectName oname = ObjectNameFactory.create("jboss.system:type=ServerConfig");
      File dir = (File)getMbeanServer().getAttribute(oname, "HomeDir");
      return dir;
    }
    catch (JMException e)
    {
      return null;
    }
  }

  public File getServerDataDir()
  {
    try
    {
      ObjectName oname = ObjectNameFactory.create("jboss.system:type=ServerConfig");
      File dir = (File)getMbeanServer().getAttribute(oname, "ServerDataDir");
      return dir;
    }
    catch (JMException e)
    {
      return null;
    }
  }
  
  public File getServerDeployDir() {
	  return(getServerDataDir());
  }

  public String getWebServiceHost()
  {
    return webServiceHost;
  }

  public void setWebServiceHost(String host) throws UnknownHostException
  {
    if (host == null || host.trim().length() == 0)
    {
      log.debug("Using undefined host: " + UNDEFINED_HOSTNAME);
      host = UNDEFINED_HOSTNAME;
    }
    if ("0.0.0.0".equals(host))
    {
      InetAddress localHost = InetAddress.getLocalHost();
      log.debug("Using local host: " + localHost.getHostName());
      host = localHost.getHostName();
    }
    this.webServiceHost = host;
  }

  public void setWebServicePort(int port)
  {
    this.webServicePort = port;
  }

  public void setWebServiceSecurePort(int port)
  {
    this.webServiceSecurePort = port;
  }

  public int getWebServicePort()
  {
    if (webServicePort <= 0)
      webServicePort = getConnectorPort("HTTP/1.1", false);

    int localPort = webServicePort;
    if (localPort <= 0)
    {
      // Do not initialize webServicePort with the default, the connector port may become available later
      log.warn("Unable to calculate 'WebServicePort', using default '8080'");
      localPort = 8080;
    }

    return localPort;
  }


  public int getWebServiceSecurePort()
  {
    if (webServiceSecurePort <= 0)
      webServiceSecurePort = getConnectorPort("HTTP/1.1", true);

    int localPort = webServiceSecurePort;
    if (localPort <= 0)
    {
      // Do not initialize webServiceSecurePort with the default, the connector port may become available later
      log.warn("Unable to calculate 'WebServiceSecurePort', using default '8443'");
      localPort = 8443;
    }

    return localPort;
  }

  private int getConnectorPort(final String protocol, final boolean secure)
  {
    int port = -1;

    try
    {
      ObjectName connectors = new ObjectName("jboss.web:type=Connector,*");

      Set connectorNames = getMbeanServer().queryNames(connectors, null);
      for (Object current : connectorNames)
      {
        ObjectName currentName = (ObjectName)current;

        try
        {
          int connectorPort = (Integer)getMbeanServer().getAttribute(currentName, "port");
          boolean connectorSecure = (Boolean)getMbeanServer().getAttribute(currentName, "secure");
          String connectorProtocol = (String)getMbeanServer().getAttribute(currentName, "protocol");


          if (protocol.equals(connectorProtocol) && secure == connectorSecure)
          {
            if (port > -1)
            {
              log.warn("Found multiple connectors for protocol='" + protocol + "' and secure='" + secure + "', using first port found '" + port + "'");
            }
            else
            {
              port = connectorPort;
            }
          }
        }
        catch (AttributeNotFoundException ignored)
        {
        }
      }

      return port;
    }
    catch (JMException e)
    {
      return -1;
    }
  }
  
  public String getClusterNodeName() {
	 if (this.clusterNodeName == null || "".equals(this.clusterNodeName.trim())) {
		 return nonClusteredServerName;
	 }
	 return this.clusterNodeName;
  }
  
  public void setClusterNodeName(String clusterNodeName) {
	  this.clusterNodeName = clusterNodeName;
  }
  
  
  public String getUddiNodeName() {
	  return this.getWebServiceHost() + "-" + this.getWebServicePort();
  }
  
}
