/*
 * Copyright (c) 2000, 2003, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package org.jboss.com.sun.corba.se.impl.protocol.giopmsgheaders;

import org.jboss.com.sun.corba.se.impl.encoding.CDRInputStream;
import org.jboss.com.sun.corba.se.impl.logging.ORBUtilSystemException;
import org.jboss.com.sun.corba.se.impl.orbutil.ORBUtility;
import org.jboss.com.sun.corba.se.spi.ior.IOR;
import org.jboss.com.sun.corba.se.spi.ior.IORFactories;
import org.jboss.com.sun.corba.se.spi.ior.iiop.GIOPVersion;
import org.jboss.com.sun.corba.se.spi.logging.CORBALogDomains;
import org.jboss.com.sun.corba.se.spi.orb.ORB;
import org.omg.CORBA.CompletionStatus;
import org.omg.CORBA.SystemException;

/**
 * This implements the GIOP 1.2 LocateReply header.
 * 
 * @author Ram Jeyaraman 05/14/2000
 */

public final class LocateReplyMessage_1_2 extends Message_1_2 implements LocateReplyMessage
{

    // Instance variables

    private ORBUtilSystemException wrapper = null;

    private int reply_status = 0;

    private IOR ior = null;

    private String exClassName = null;

    private int minorCode = 0;

    private CompletionStatus completionStatus = null;

    private short addrDisposition = KeyAddr.value; // default;

    // Constructors

    LocateReplyMessage_1_2(ORB orb)
    {
        this.wrapper = ORBUtilSystemException.get(orb, CORBALogDomains.RPC_PROTOCOL);
    }

    LocateReplyMessage_1_2(ORB orb, int _request_id, int _reply_status, IOR _ior)
    {
        super(Message.GIOPBigMagic, GIOPVersion.V1_2, FLAG_NO_FRAG_BIG_ENDIAN, Message.GIOPLocateReply, 0);
        this.wrapper = ORBUtilSystemException.get(orb, CORBALogDomains.RPC_PROTOCOL);
        request_id = _request_id;
        reply_status = _reply_status;
        ior = _ior;
    }

    // Accessor methods

    public int getRequestId()
    {
        return this.request_id;
    }

    public int getReplyStatus()
    {
        return this.reply_status;
    }

    public short getAddrDisposition()
    {
        return this.addrDisposition;
    }

    public SystemException getSystemException(String message)
    {
        return MessageBase.getSystemException(exClassName, minorCode, completionStatus, message, wrapper);
    }

    public IOR getIOR()
    {
        return this.ior;
    }

    // IO methods

    public void read(org.omg.CORBA.portable.InputStream istream)
    {
        super.read(istream);
        this.request_id = istream.read_ulong();
        this.reply_status = istream.read_long();
        isValidReplyStatus(this.reply_status); // raises exception on error

        // GIOP 1.2 LocateReply message bodies are not aligned on 8 byte boundaries.

        // The code below reads the reply body in some cases LOC_SYSTEM_EXCEPTION & OBJECT_FORWARD & OBJECT_FORWARD_PERM
        // & LOC_NEEDS_ADDRESSING_MODE
        if (this.reply_status == LOC_SYSTEM_EXCEPTION)
        {

            String reposId = istream.read_string();
            this.exClassName = ORBUtility.classNameOf(reposId);
            this.minorCode = istream.read_long();
            int status = istream.read_long();

            switch (status)
            {
                case CompletionStatus._COMPLETED_YES :
                    this.completionStatus = CompletionStatus.COMPLETED_YES;
                    break;
                case CompletionStatus._COMPLETED_NO :
                    this.completionStatus = CompletionStatus.COMPLETED_NO;
                    break;
                case CompletionStatus._COMPLETED_MAYBE :
                    this.completionStatus = CompletionStatus.COMPLETED_MAYBE;
                    break;
                default :
                    throw wrapper.badCompletionStatusInLocateReply(CompletionStatus.COMPLETED_MAYBE,
                            new Integer(status));
            }
        }
        else if ((this.reply_status == OBJECT_FORWARD) || (this.reply_status == OBJECT_FORWARD_PERM))
        {
            CDRInputStream cdr = (CDRInputStream) istream;
            this.ior = IORFactories.makeIOR(cdr);
        }
        else if (this.reply_status == LOC_NEEDS_ADDRESSING_MODE)
        {
            // read GIOP::AddressingDisposition from body and resend the original request using the requested addressing
            // mode. The resending is transparent to the caller.
            this.addrDisposition = AddressingDispositionHelper.read(istream);
        }
    }

    // Note, this writes only the header information. SystemException or IOR or GIOP::AddressingDisposition may be
    // written afterwards into the reply mesg body.
    public void write(org.omg.CORBA.portable.OutputStream ostream)
    {
        super.write(ostream);
        ostream.write_ulong(this.request_id);
        ostream.write_long(this.reply_status);

        // GIOP 1.2 LocateReply message bodies are not aligned on 8 byte boundaries.
    }

    // Static methods

    public static void isValidReplyStatus(int replyStatus)
    {
        switch (replyStatus)
        {
            case UNKNOWN_OBJECT :
            case OBJECT_HERE :
            case OBJECT_FORWARD :
            case OBJECT_FORWARD_PERM :
            case LOC_SYSTEM_EXCEPTION :
            case LOC_NEEDS_ADDRESSING_MODE :
                break;
            default :
                ORBUtilSystemException localWrapper = ORBUtilSystemException.get(CORBALogDomains.RPC_PROTOCOL);
                throw localWrapper.illegalReplyStatus(CompletionStatus.COMPLETED_MAYBE);
        }
    }

    public void callback(MessageHandler handler) throws java.io.IOException
    {
        handler.handleInput(this);
    }
} // class LocateReplyMessage_1_2
