/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.switchboard.jbmeta.javaee.environment;

import java.util.Properties;

import org.jboss.metadata.javaee.spec.DataSourceMetaData;
import org.jboss.metadata.javaee.spec.IsolationLevelType;
import org.jboss.metadata.javaee.spec.PropertiesMetaData;
import org.jboss.metadata.javaee.spec.PropertyMetaData;
import org.jboss.switchboard.javaee.environment.ConnectionIsolationLevel;
import org.jboss.switchboard.javaee.environment.DataSourceType;
import org.jboss.switchboard.javaee.util.EnvironmentEntryUtil;

/**
 * DataSourceReference
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class DataSourceReference implements DataSourceType
{

   private DataSourceMetaData delegate;
   
   private Properties properties = new Properties(); 
   
   public DataSourceReference(DataSourceMetaData datasourceMetadata)
   {
      if (datasourceMetadata == null)
      {
         throw new IllegalArgumentException("DataSourceMetaData cannot be null while creating " + DataSourceReference.class.getName());
      }
      this.delegate = datasourceMetadata;
      
      // init the properties
      this.initProperties();
   }
   
   @Override
   public String getClassName()
   {
      return this.delegate.getClassName();
   }

   @Override
   public String getDatabaseName()
   {
      return this.delegate.getDatabaseName();
   }

   @Override
   public int getInitialPoolSize()
   {
      return this.delegate.getInitialPoolSize();
   }

   @Override
   public ConnectionIsolationLevel getIsolationLevel()
   {
      IsolationLevelType isolation = this.delegate.getIsolationLevel();
      if (isolation == null)
      {
         return null;
      }
      switch(isolation)
      {
         case TRANSACTION_READ_COMMITTED:
            return ConnectionIsolationLevel.TRANSACTION_READ_COMMITTED;
            
         case TRANSACTION_READ_UNCOMMITTED:
            return ConnectionIsolationLevel.TRANSACTION_READ_UNCOMMITTED;
            
         case TRANSACTION_REPEATABLE_READ:
            return ConnectionIsolationLevel.TRANSACTION_REPEATABLE_READ;
            
         case TRANSACTION_SERIALIZABLE:
            return ConnectionIsolationLevel.TRANSACTION_SERIALIZABLE;
      }
      throw new IllegalArgumentException("Unrecognized isolation level: " + isolation.name() + " for datasource " + this.delegate.getName());
   }

   @Override
   public int getLoginTimeout()
   {
      return this.delegate.getLoginTimeout();
   }

   @Override
   public int getMaxIdleTime()
   {
      return this.delegate.getMaxIdleTime();
   }

   @Override
   public int getMaxPoolSize()
   {
      return this.delegate.getMaxPoolSize();
   }

   @Override
   public int getMaxStatements()
   {
      return this.delegate.getMaxStatements();
   }

   @Override
   public int getMinPoolSize()
   {
      return this.delegate.getMinPoolSize();
   }

   @Override
   public String getPassword()
   {
      return this.delegate.getPassword();
   }

   @Override
   public int getPort()
   {
      return this.delegate.getPortNumber();
   }

   @Override
   public Properties getProperties()
   {
      return this.properties;
   }

   @Override
   public String getServerName()
   {
      return this.delegate.getServerName();
   }

   @Override
   public String getURL()
   {
      return this.delegate.getUrl();
   }

   @Override
   public String getUserName()
   {
      return this.delegate.getUser();
   }

   @Override
   public boolean isTransactional()
   {
      return this.delegate.isTransactional();
   }

   @Override
   public String getName()
   {
      return EnvironmentEntryUtil.getENCName(this.delegate.getName());
   }

   
   private void initProperties()
   {
      PropertiesMetaData props =  this.delegate.getProperties();
      if (props == null)
      {
         return;
      }
      for (PropertyMetaData prop : props)
      {
         if (prop == null)
         {
            continue;
         }
         this.properties.put(prop.getKey(), prop.getValue());
      }
   }
}
