/*
 * JBoss, Home of Professional Open Source.
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

package com.metamatrix.vdb.materialization;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * This class modifies the DDL, which generated of the VDB. Sometimes like in the
 * Materialization Model, we need to add extra specific DDL statements into the
 * original DDL, which are not generated by default. This class used as 
 * Intercepter after the generation, before saving, so that proper DDL can be 
 * generated for any purpose.
 */
public class ScriptDecorator {

    /**
     * Add the custom extensions or modify the original DDL script
     * into a customized format based on the database type supplied.
     * 
     * @param dbType - Database Style
     * @param orignalContent - Byte stream of original DDL
     * @return modified DDL, if custom extnsion needed, otherwise 
     * original byte stream returned
     */
    public byte[] modifyDDL(String name, byte[] orignalContent) {
         DatabaseDialect dbDialect = DatabaseDialect.getDatabaseDialectByDDLName(name);

         // For now just use simplified code to do the job fast, if we ever 
         // need we can subclass and go more fancy.
         if (dbDialect == DatabaseDialect.DB2) {
             String orginalDDL = new String(orignalContent);
             try {
                 Pattern pattern = Pattern.compile("\\)%"); //$NON-NLS-1$
                 Matcher matcher = pattern.matcher(orginalDDL);
                 String modifiedDDL = matcher.replaceAll(") NOT LOGGED INITIALLY%"); //$NON-NLS-1$
                 return modifiedDDL.getBytes();
            } catch (PatternSyntaxException e) {
                // Ignore as this only happends on bad expression                
            }
         }
         return orignalContent;
    }
    
    
}
