/*
 * JBoss, Home of Professional Open Source.
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

package com.metamatrix.console.ui.views.connectorbinding;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumn;

import com.metamatrix.common.config.api.Configuration;
import com.metamatrix.common.config.api.ProductServiceConfig;
import com.metamatrix.common.log.LogManager;
import com.metamatrix.console.models.ConnectorManager;
import com.metamatrix.console.ui.util.BasicWizardSubpanelContainer;
import com.metamatrix.console.ui.util.WizardInterface;
import com.metamatrix.console.ui.views.deploy.util.DeployPkgUtils;
import com.metamatrix.console.ui.views.deploy.util.DeployTableSorter;
import com.metamatrix.console.util.ExceptionUtility;
import com.metamatrix.console.util.LogContexts;
import com.metamatrix.toolbox.ui.widget.CheckBox;
import com.metamatrix.toolbox.ui.widget.TableWidget;
import com.metamatrix.toolbox.ui.widget.TitledBorder;
import com.metamatrix.toolbox.ui.widget.table.DefaultTableModel;
import com.metamatrix.toolbox.ui.widget.table.EnhancedTableColumn;
import com.metamatrix.toolbox.ui.widget.table.EnhancedTableColumnModel;

// ===

public class ImportBindingWizardPSCEnablePanel
        extends BasicWizardSubpanelContainer
     implements ActionListener,
                ListSelectionListener,
                TableModelListener {
    ///////////////////////////////////////////////////////////////////////////
    // CONSTANTS
    ///////////////////////////////////////////////////////////////////////////

//    private int TEXTAREA_MAXLENGTH      = 255;
    
    
    private static /*final*/ String[] SERVICE_HDRS;
    private static final int PSC_COL                    = 0;
    private static final int ENABLED_COL                = 1;


    private static /*final*/ String[] BINDING_HDRS;
    private static final int BINDING_COL                = 0;
    
    private static final String STEP_DESC="Set Enabled State for each PSC to deploy the Binding(s) to and select Finish to Create the Binding(s)."; //$NON-NLS-1$

    

    ///////////////////////////////////////////////////////////////////////////
    // INITIALIZER
    ///////////////////////////////////////////////////////////////////////////

    static {
        SERVICE_HDRS = new String[2];
        SERVICE_HDRS[PSC_COL]               = "PSC"; //$NON-NLS-1$
        SERVICE_HDRS[ENABLED_COL]           = "Enabled"; //$NON-NLS-1$

        BINDING_HDRS = new String[1];
        BINDING_HDRS[BINDING_COL]               = "PSC"; //$NON-NLS-1$

    }

    ///////////////////////////////////////////////////////////////////////////
    // CONTROLS
    ///////////////////////////////////////////////////////////////////////////

	private TableWidget tblPSCs;

    ///////////////////////////////////////////////////////////////////////////
    // FIELDS
    ///////////////////////////////////////////////////////////////////////////

	private DefaultTableModel tmdlPSCs;

	// Next Startup PSC Map:  Map PSC to enabled state
    private HashMap mapPSCtoEnabledForNextStartup           = new HashMap();

	// PSC Name to Next Startup PSC Xref
    private HashMap mapNextStartupPSCXref            = new HashMap();

    private int numRowsDifferentNextStartup = 0;
    private CheckBox chk; // the table cell editor component for the enabled col
    
    private TableWidget tblBindings;

    private DefaultTableModel tmdlBindings;
    
    private JPanel pnlOuter                     = new JPanel();

	private JPanel pnlTable         = new JPanel();

	private ConnectorManager connectorManager;

    public ImportBindingWizardPSCEnablePanel(WizardInterface wizardInterface, ConnectorManager connecMgr) {
        super(wizardInterface);
        connectorManager = connecMgr;
        init();
    }

    public void setNewConnectorBindingInfo(Collection bindingNames) {

       populateTable(bindingNames);

	}

    private void init() {
        createTablePanel();

        pnlOuter.setLayout(new GridBagLayout());

        JPanel bTablePanel = createBindingsTable();
                
        pnlOuter.add(bTablePanel, new GridBagConstraints(0, 0, 1, 1, 1.0, 0.5, 
             GridBagConstraints.NORTH, GridBagConstraints.BOTH, 
             new Insets(2, 0, 2, 0), 0, 0));
        
        
        pnlOuter.add(pnlTable, new GridBagConstraints(0, 2, 
            GridBagConstraints.REMAINDER, GridBagConstraints.REMAINDER, 
            1.0, 1.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, 
            new Insets(12, 3, 3, 3), 0, 0));
        

        setMainContent(pnlOuter);
        setStepText(3, STEP_DESC);
		populateTable();
	}

    private JPanel createTablePanel() {
        pnlTable = new JPanel();
        pnlTable.setLayout(new GridLayout(1, 1));        
        tblPSCs = new TableWidget();

        tmdlPSCs = DeployPkgUtils.setup(
                tblPSCs,
                SERVICE_HDRS,
                10,
                new int[] {ENABLED_COL});
        tblPSCs.getSelectionModel().addListSelectionListener(this);
        tblPSCs.setComparator(new DeployTableSorter());

        doTableSetup();
        
        tmdlPSCs.addTableModelListener(this);

		JScrollPane spnServices = new JScrollPane(tblPSCs);
        pnlTable.add(spnServices);

        return pnlTable;
    }
    
    private JPanel createBindingsTable() {
        JPanel bindingTable = new JPanel();
        bindingTable.setLayout(new GridLayout(1, 1));        
        tblBindings = new TableWidget();
        tblBindings.setComparator(new DeployTableSorter()); 
        
        tmdlBindings = DeployPkgUtils.setup(
                             tblBindings,
                             BINDING_HDRS,
                             5,
                             new int[] {});  
        
        tmdlBindings.setEditable(false);
 
        JScrollPane spnServices = new JScrollPane(tblBindings);
        bindingTable.add(spnServices);
        
        TitledBorder border;
        String title = "Binding(s) to import"; //$NON-NLS-1$
        border = new TitledBorder(title);
       

        bindingTable.setBorder(border);
        return bindingTable;
    }
    

    private void populateTable() {
        loadPSCsForNextStartup();

        createPSCXrefs();

        tmdlPSCs.setNumRows(0);
        try {

            // Process Next Startup set
            if (mapPSCtoEnabledForNextStartup != null) {
                Iterator itPsc
                    = mapPSCtoEnabledForNextStartup.keySet().iterator();

                // drive the process by walking the NextStartup hashmap
                while (itPsc.hasNext()) {
                    ProductServiceConfig psc = (ProductServiceConfig)itPsc.next();

                    Vector row = new Vector(SERVICE_HDRS.length);
                    row.setSize(SERVICE_HDRS.length);

                    row.setElementAt(psc, PSC_COL);
                   
					Boolean enabledForNextStartup = 
							(Boolean)mapPSCtoEnabledForNextStartup.get(psc);
					
                    row.setElementAt(enabledForNextStartup, ENABLED_COL);

                    tmdlPSCs.addRow(row);
                }
            }
		} catch (Exception theException) {
            ExceptionUtility.showMessage("  ", //$NON-NLS-1$
                //getString("msg.configmgrproblem",
                //          new Object[] {getClass(), "setDomainObject"}),
                theException);
            LogManager.logError(
                LogContexts.CONNECTOR_BINDINGS,
                theException,
                getClass() + ":setDomainObject"); //$NON-NLS-1$
        }
	}
    
    private void populateTable(Collection bindings) {
        tmdlBindings.setRowCount(0);
        try {

            // Process Next Startup set
                Iterator itBinding = bindings.iterator();

                // drive the process by walking the NextStartup hashmap
                while (itBinding.hasNext()) {
                    String binding =
                        (String)itBinding.next();

                    Vector row = new Vector(BINDING_HDRS.length);
                    row.setSize(BINDING_HDRS.length);

                    row.setElementAt(binding, BINDING_COL);
                    
                    tmdlBindings.addRow(row);                    
                    
                }
        } catch (Exception theException) {
            ExceptionUtility.showMessage("  ", //$NON-NLS-1$
                //getString("msg.configmgrproblem",
                //          new Object[] {getClass(), "setDomainObject"}),
                theException);
            LogManager.logError(
                LogContexts.CONNECTOR_BINDINGS,
                theException,
                getClass() + ":setDomainObject"); //$NON-NLS-1$
        }
    }

    

    private void doTableSetup() {
        // customize the table
        tblPSCs.sizeColumnsToFitData();

        // fix column for Next Startup
        EnhancedTableColumnModel etcmNextStartup = 
        		tblPSCs.getEnhancedColumnModel();
        TableColumn clmConnBindColumnNextStartup = 
        		etcmNextStartup.getColumn(ENABLED_COL);
        tblPSCs.sizeColumnToFitData(
        		(EnhancedTableColumn)clmConnBindColumnNextStartup);
		sortPSCTable();
    }

    private void sortPSCTable() {
        EnhancedTableColumnModel etcmPSCs = tblPSCs.getEnhancedColumnModel();
        TableColumn clmPSCColumn = etcmPSCs.getColumn(PSC_COL);
        etcmPSCs
            .setColumnSortedAscending((EnhancedTableColumn)clmPSCColumn, false);
    }

	private void loadPSCsForNextStartup() {
		mapPSCtoEnabledForNextStartup.clear();
        try {
        	Configuration config = getNextStartupConfig();
            Collection colPsc = connectorManager.getAllConnectorsPSCsByConfig(
            		config);

            if (colPsc != null) {
                Iterator itPsc = colPsc.iterator();
                while (itPsc.hasNext()) {
                    ProductServiceConfig psc =
                        (ProductServiceConfig)itPsc.next();

					Boolean enabled = null;
                    if (psc.getServiceComponentDefnIDs().size() > 0) {
               			enabled = Boolean.TRUE;
               		} else {
               			enabled = Boolean.FALSE;	
               		}
					mapPSCtoEnabledForNextStartup.put(psc, enabled);
				}
            }
        } catch (Exception theException) {
            ExceptionUtility.showMessage("  ", //$NON-NLS-1$
                theException);
            LogManager.logError(LogContexts.CONNECTOR_BINDINGS, theException,
            		getClass() + ":setDomainObject"); //$NON-NLS-1$
        }
	}

	private void createPSCXrefs() {
    	// Next Startup
        Iterator itPscs = mapPSCtoEnabledForNextStartup.keySet().iterator();
        while(itPscs.hasNext()) {
            ProductServiceConfig psc = (ProductServiceConfig)itPscs.next();
            mapNextStartupPSCXref.put(psc.getName(), psc);
        }
	}

	public void actionPerformed(ActionEvent theEvent) {
    }

    public void tableChanged(TableModelEvent theEvent) {
            int row = theEvent.getFirstRow();
            ProductServiceConfig psc
                = (ProductServiceConfig)tmdlPSCs.getValueAt(row, PSC_COL);

            // get the proper PSC for NextStartup
            ProductServiceConfig pscNS
                = (ProductServiceConfig)
                        mapNextStartupPSCXref.get(psc.getName());

            Object saveEnabled = mapPSCtoEnabledForNextStartup.get(pscNS);

			if (!saveEnabled.equals(tmdlPSCs.getValueAt(row, ENABLED_COL))) {
                numRowsDifferentNextStartup++;
            } else {
                if (numRowsDifferentNextStartup > 0) {
                    numRowsDifferentNextStartup--;
                }
            }
            checkResetState();
	}

    public void valueChanged(ListSelectionEvent theEvent) {
        // done one time to setup the checkbox action listener
        int row = tblPSCs.getSelectedRow();
        if (row != -1) {
            TableCellEditor editor = tblPSCs.getCellEditor(row, ENABLED_COL);
            chk = (CheckBox)editor.getTableCellEditorComponent(
                    tblPSCs,
                    tblPSCs.getValueAt(row, ENABLED_COL),
                    true,
                    row,
                    ENABLED_COL);
            chk.addActionListener(this);
            tblPSCs.getSelectionModel().removeListSelectionListener(this);
        }
    }

	private void checkResetState() {
    }

	private Configuration getNextStartupConfig() {
        Configuration cfg =  null;
        try {
            cfg = connectorManager.getNextStartupConfig();
        } catch (Exception e) {
            ExceptionUtility.showMessage(
            		"Failed retrieving the Next Startup Config", e); //$NON-NLS-1$
        }
        return cfg;
    }
    
    public ProductServiceConfig[] getEnabledConfigs() {
    	int numRows = tblPSCs.getRowCount();
    	java.util.List pscsList = new ArrayList(numRows);
    	for (int i = 0; i < numRows; i++) {
    		Boolean enabledBool = (Boolean)tblPSCs.getModel().getValueAt(i,
    				ENABLED_COL);
    		if (enabledBool.booleanValue()) {
    			ProductServiceConfig psc = 
    					(ProductServiceConfig)tblPSCs.getModel().getValueAt(i,
    					PSC_COL);
    			pscsList.add(psc);
    		}
    	}
    	ProductServiceConfig[] pscs = 
    			new ProductServiceConfig[pscsList.size()];
    	Iterator it = pscsList.iterator();
    	for (int i = 0; it.hasNext(); i++) {
    		pscs[i] = (ProductServiceConfig)it.next();
    	}
    	return pscs;
    }
}