/*
 * JBoss, Home of Professional Open Source.
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

package org.teiid.query.sql.visitor;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.teiid.api.exception.query.QueryValidatorException;
import org.teiid.core.TeiidRuntimeException;
import org.teiid.core.util.Assertion;
import org.teiid.query.QueryPlugin;
import org.teiid.query.sql.lang.BetweenCriteria;
import org.teiid.query.sql.lang.CompareCriteria;
import org.teiid.query.sql.lang.Criteria;
import org.teiid.query.sql.lang.DependentSetCriteria;
import org.teiid.query.sql.lang.IsNullCriteria;
import org.teiid.query.sql.lang.MatchCriteria;
import org.teiid.query.sql.lang.SetCriteria;
import org.teiid.query.sql.navigator.PreOrderNavigator;
import org.teiid.query.sql.proc.CriteriaSelector;
import org.teiid.query.sql.symbol.ElementSymbol;
import org.teiid.query.sql.symbol.Expression;
import org.teiid.query.sql.symbol.Reference;


/**
 * <p> This class is used to translate criteria specified on the user's update command against
 * the virtual group, the elements on this criteria are replaced by elements on the query
 * transformation that defines the virtual group. Parts of the criteria are selectively translated
 * if a CriteriaSelector is specified, also if the user explicty defines translations for some
 * of the elements those translations override any symbol mappings.</p>
 */
public class CriteriaTranslatorVisitor extends ExpressionMappingVisitor {
	
	class CriteriaTranslatorNavigator extends PreOrderNavigator {

		public CriteriaTranslatorNavigator() {
			super(CriteriaTranslatorVisitor.this);
		}
		
	    /**
	     * <p> This method updates the <code>BetweenCriteria</code> object it receives as an
	     * argument by replacing the virtual elements present in the expressions in the
	     * function with translated expressions.</p>
	     * @param obj The BetweenCriteria object to be updated with translated expressions
	     */
	    public void visit(BetweenCriteria obj) {
	        if (!selectorContainsCriteriaElements(obj, CriteriaSelector.BETWEEN)) {
	        	throw new TeiidRuntimeException(new QueryValidatorException(QueryPlugin.Util.getString("Translate.error", obj, selector))); //$NON-NLS-1$ 
	        }
	        super.visit(obj);
	    }
	    
	    /**
	     * <p> This method updates the <code>CompareCriteria</code> object it receives as an
	     * argument by replacing the virtual elements present in the expressions in the
	     * function with translated expressions.</p>
	     * @param obj The CompareCriteria object to be updated with translated expressions
	     */
	    public void visit(CompareCriteria obj) {
	        
	        if (!selectorContainsCriteriaElements(obj, obj.getOperator())) {
	        	throw new TeiidRuntimeException(new QueryValidatorException(QueryPlugin.Util.getString("Translate.error", obj, selector))); //$NON-NLS-1$ 
	        }

	        super.visit(obj);
	    }

	    /**
	     * <p> This method updates the <code>IsNullCriteria</code> object it receives as an
	     * argument by replacing the virtual elements present in the expressions in the
	     * function with translated expressions.</p>
	     * @param obj The IsNullCriteria object to be updated with translated expressions
	     */
	    public void visit(IsNullCriteria obj) {

	        if (!selectorContainsCriteriaElements(obj, CriteriaSelector.IS_NULL)) {
	        	throw new TeiidRuntimeException(new QueryValidatorException(QueryPlugin.Util.getString("Translate.error", obj, selector))); //$NON-NLS-1$ 
	        }
	        super.visit(obj);
	    }

	    /**
	     * <p> This method updates the <code>MatchCriteria</code> object it receives as an
	     * argument by replacing the virtual elements present in the expressions in the
	     * function with translated expressions</p>
	     * @param obj The SetCriteria object to be updated with translated expressions
	     */
	    public void visit(MatchCriteria obj) {
	        
	        if (!selectorContainsCriteriaElements(obj, CriteriaSelector.LIKE)) {
	        	throw new TeiidRuntimeException(new QueryValidatorException(QueryPlugin.Util.getString("Translate.error", obj, selector))); //$NON-NLS-1$ 
	        }

	        super.visit(obj);
	    }
	    
	    /**
	     * <p> This method updates the <code>SetCriteria</code> object it receives as an
	     * argument by replacing the virtual elements present in the expressions in the
	     * function with translated expressions</p>
	     * @param obj The SetCriteria object to be updated with translated expressions
	     */
	    public void visit(SetCriteria obj) {
	        
	        if (!selectorContainsCriteriaElements(obj, CriteriaSelector.IN)) {
	        	throw new TeiidRuntimeException(new QueryValidatorException(QueryPlugin.Util.getString("Translate.error", obj, selector))); //$NON-NLS-1$
	        }
	        
	        super.visit(obj);
	    }

	    /**
	     * <p> This method updates the <code>SetCriteria</code> object it receives as an
	     * argument by replacing the virtual elements present in the expressions in the
	     * function with translated expressions</p>
	     * @param obj The SetCriteria object to be updated with translated expressions
	     */
	    public void visit(DependentSetCriteria obj) {
	        
	        if (!selectorContainsCriteriaElements(obj, CriteriaSelector.IN)) {
	        	throw new TeiidRuntimeException(new QueryValidatorException(QueryPlugin.Util.getString("Translate.error", obj, selector))); //$NON-NLS-1$
	        }
	        
	        super.visit(obj);
	    }
		
	}

	// criteria selector specified on the TranslateCriteria obj
	private CriteriaSelector selector;

	// translation in for of CompareCriteria objs on the TranslateCriteria obj
	private Collection translations;

	private Map<ElementSymbol, Reference> implicitParams = new HashMap<ElementSymbol, Reference>();

    /**
     * <p> This constructor initialises the visitor</p>
     */
    public CriteriaTranslatorVisitor() {
    	this(null);
    }

    /**
     * <p> This constructor initializes this object by setting the symbolMap.</p>
     * @param symbolMap A map of virtual elements to their counterparts in transform
     * defining the virtual group
     */
    public CriteriaTranslatorVisitor(Map symbolMap) {
        super(symbolMap);
        Assertion.isNotNull(symbolMap);
    }

	/**
	 * <p>Set the criteria selector used to restrict the part of the criteria that needs to be
	 * translated.</p>
	 * @param selector The <code>CriteriaSelector</code> on the <code>TranslateCriteria</code>
	 * object
	 */
    public void setCriteriaSelector(CriteriaSelector selector) {
    	this.selector = selector;
    }

	/**
	 * <p> Set the translations to be used to replace elements on the user's command against
	 * the virtual group.</p>
     * @param translations Collection of <code>ComapreCriteria</code> objects used to
     * specify translations
     */
    public void setTranslations(Collection translations) {
    	this.translations = translations;
    }

    /* ############### Helper Methods ##################   */    
    
    private boolean selectorContainsCriteriaElements(Criteria criteria, int criteriaType) {
        int selectorType = selector.getSelectorType();
        if(selectorType!= CriteriaSelector.NO_TYPE && selectorType != criteriaType) {
            return false;
        } else if(selector.hasElements()) {                
            Iterator selectElmnIter = selector.getElements().iterator();
            Collection<ElementSymbol> critElmnts = ElementCollectorVisitor.getElements(criteria, true);
            while(selectElmnIter.hasNext()) {
                ElementSymbol selectElmnt = (ElementSymbol) selectElmnIter.next();
                if(critElmnts.contains(selectElmnt)) {
                    return true;
                }
            }
            return false;
        }
        return true;
    }
    
    @Override
    public Expression replaceExpression(Expression obj) {
    	if (this.translations != null && obj instanceof ElementSymbol) {
			Iterator transIter = this.translations.iterator();
			while(transIter.hasNext()) {
				CompareCriteria compCrit = (CompareCriteria) transIter.next();
				Collection<ElementSymbol> leftElmnts = ElementCollectorVisitor.getElements(compCrit.getLeftExpression(), true);
				// there is always only one element
				ElementSymbol element = leftElmnts.iterator().next();
				if(obj.equals(element)) {
					return compCrit.getRightExpression();
				}
			}
     	}
    	/*
    	 * Special handling for references in translated criteria.
    	 * We need to create a locally valid reference name.
    	 */
    	if (obj instanceof Reference) {
    		Reference implicit = (Reference)obj;
    		ElementSymbol key = null;
    		if (implicit.isPositional()) {
    			key = new ElementSymbol("$INPUT." + implicit.getContextSymbol()); //$NON-NLS-1$
    		} else {
    			key = new ElementSymbol("$INPUT." + implicit.getExpression().getName()); //$NON-NLS-1$
    		}
    		key.setType(implicit.getType());
    		this.implicitParams.put(key, implicit);
    		return new Reference(key);
    	}
    	return super.replaceExpression(obj);
    }

    public Map<ElementSymbol, Reference> getImplicitParams() {
		return implicitParams;
	}
    
    public void translate(Criteria crit) throws QueryValidatorException {
    	CriteriaTranslatorNavigator nav = new CriteriaTranslatorNavigator();
    	try {
    		crit.acceptVisitor(nav);
    	} catch (TeiidRuntimeException e) {
    		throw (QueryValidatorException)e.getCause();
    	}
    }
    
}
