package org.jboss.testharness.spi.helpers;

import static org.jboss.testharness.api.util.LogUtil.logger;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;

import org.jboss.testharness.api.Configurable;
import org.jboss.testharness.api.Configuration;
import org.jboss.testharness.api.DeploymentException;
import org.jboss.testharness.api.DeploymentExceptionTransformer;
import org.jboss.testharness.properties.PropertiesManager;

public abstract class AbstractContainerConnector implements Configurable
{

   public static String JAVA_OPTS = " -ea";

   public static final String JAVA_OPTS_PROPERTY_NAME = "org.jboss.testharness.container.javaOpts";

   private static final String BOOT_TIMEOUT_PROPERTY_NAME = "org.jboss.testharness.container.bootTimeout";
   public static final String FORCE_RESTART_PROPERTY_NAME = "org.jboss.testharness.container.forceRestart";
   public static final String HOST_PROPERTY_NAME = "org.jboss.testharness.container.host";
   public static final String PORT_PROPERTY_NAME = "org.jboss.testharness.container.port";
   public static final String SHUTDOWN_DELAY_PROPERTY_NAME = "org.jboss.testharness.container.shutdownDelay";

   private static final String DEPLOYMENT_EXCEPTION_TRANSFORMER_PROPERTY_NAME = "org.jboss.testharness.container.deploymentExceptionTransformer";

   public static final String EXTRA_CONFIGURATION_DIR_PROPERTY_NAME = "org.jboss.testharness.container.extraConfigurationDir";

   private final PropertiesManager properties;
   private Configuration configuration;
   private boolean wasStarted;

   private String serverDirectory;
   private String httpUrl;
   private Long bootTimeout;
   private String javaOpts;
   private Boolean forceRestart;
   private Integer shutdownDelay;
   private String host;
   private String port;

   private DeploymentExceptionTransformer deploymentExceptionTransformer;

   private String extraConfigurationDir;

   protected static void copy(InputStream inputStream, File file) throws IOException
   {
      OutputStream os = new FileOutputStream(file);
      try
      {
         copy(inputStream, os);
      }
      finally
      {
         os.close();
      }
   }

   public static void copy(InputStream source, OutputStream destination) throws IOException
   {
      if (source == null)
      {
         throw new IllegalArgumentException("source cannot be null");
      }
      if (destination == null)
      {
         throw new IllegalArgumentException("destination cannot be null");
      }
      byte[] readBuffer = new byte[2156]; 
      int bytesIn = 0; 
      while((bytesIn = source.read(readBuffer)) != -1) 
      { 
         destination.write(readBuffer, 0, bytesIn); 
      }
   }

   public AbstractContainerConnector() throws IOException
   {
      this.properties = new PropertiesManager();
      String extraConfigurationDir = getExtraConfigurationDir();
      if (extraConfigurationDir != null)
      {
         File extraConfiguration = new File(extraConfigurationDir);
         if (extraConfiguration.isDirectory())
         {
            File buildProperties = new File(extraConfiguration, "build.properties");
            if (buildProperties.exists())
            {
               loadProperties(buildProperties);
            }
            File localBuildProperties = new File(extraConfiguration, "local.build.properties");
            if (localBuildProperties.exists())
            {
               loadProperties(localBuildProperties);
            }
         }
      }
   }

   public void setConfiguration(Configuration configuration)
   {
      this.configuration = configuration;
   }

   /**
    * Check that the server is up by attempting to connect to it's front page
    * @return true if the connection was made
    */
   protected boolean isServerUp()
   {
      try
      {
         URLConnection connection = new URL(getHttpUrl()).openConnection();
         if (!(connection instanceof HttpURLConnection))
         {
            throw new IllegalStateException("Not an http connection! " + connection);
         }
         HttpURLConnection httpConnection = (HttpURLConnection) connection;
         httpConnection.connect();
         if (httpConnection.getResponseCode() != HttpURLConnection.HTTP_OK)
         {
            return false;
         }
      }
      catch (Exception e)
      {
         return false;
      }
      logger().info("Connected to server over http");
      return true;
   }

   public void setup() throws IOException
   {
      configuration.setHost(getHost() + ":" + getPort());
      logger().info("Using server " + getServerDirectory() + " (" + getHttpUrl() + ")");
      restartServer();
   }

   protected void restartServer() throws IOException
   {
      if (getForceRestart())
      {
         if (isServerUp())
         {
            logger().info("Shutting down server as in force-restart mode");
            shutdownServer();
            try
            {
               Thread.sleep(getShutdownDelay());
            }
            catch (InterruptedException e)
            {
               Thread.currentThread().interrupt();
            }
         }
      }
      if (!isServerUp())
      {
         wasStarted = true;
         startServer();
         logger().info("Starting server");
         // Wait for server to come up
         long timeoutTime = System.currentTimeMillis() + getServerBootTimeout();
         boolean interrupted = false;
         while (timeoutTime > System.currentTimeMillis())
         {
            if (isServerUp())
            {
               logger().info("Started server");
               return;
            }
            try
            {
               Thread.sleep(200);
            }
            catch (InterruptedException e)
            {
               interrupted = true;
            }
         }
         if (interrupted)
         {
            Thread.currentThread().interrupt();
         }
         // If we got this far something went wrong
         logger().warning("Unable to connect to server after " + getServerBootTimeout() + "ms, giving up!");
         shutdownServer();
         throw new IllegalStateException("Error connecting to server");
      }
   }

   protected void loadProperties(File file) throws IOException
   {
      InputStream is = new FileInputStream(file);
      try
      {
         System.getProperties().load(is);
      }
      finally
      {
         is.close();
      }
   }

   public void cleanup() throws IOException
   {
      if (wasStarted)
      {
         logger().info("Shutting down server");
         shutdownServer();
      }
   }

   protected abstract void shutdownServer() throws IOException;

   protected abstract void startServer() throws IOException; 

   protected void launch(String directory, String scriptFileName, String params) throws IOException
   {
      String osName = System.getProperty("os.name");
      Runtime runtime = Runtime.getRuntime();

      directory = new File(directory).getPath();

      Process p = null;
      if (osName.startsWith("Windows"))
      {
         String command[] = {
               "cmd.exe",
               "/C",
               "set JAVA_OPTS=" + getJavaOpts() + " & cd /D " + directory + " & " + scriptFileName + ".bat " + params
         };
         p = runtime.exec(command);
      }
      else
      {
         String command[] = {
               "sh",
               "-c",
               "cd " + directory + ";JAVA_OPTS=\"" + getJavaOpts() + "\" ./" + scriptFileName + ".sh " + params
         };
         p = runtime.exec(command);
      }
      dump(p.getErrorStream());
      dump(p.getInputStream());
   }

   protected void dump(final InputStream is)
   {
      new Thread(new Runnable()
      {
         public void run()
         {
            try
            {
               DataOutputStream out = new DataOutputStream(new FileOutputStream(configuration.getOutputDirectory() + File.separator + getLogName()));
               int c;
               while((c = is.read()) != -1)
               {
                  out.writeByte(c);
               }
               is.close();
               out.close();
            }
            catch(IOException e)
            {
               System.err.println("Error Writing/Reading Streams.");
            }
         }
      }).start();
   }

   protected abstract String getServerHomePropertyName();

   protected String getServerDirectory()
   {
      if (serverDirectory == null)
      {
         serverDirectory = new File(getProperties().getStringValue(getServerHomePropertyName(), null, true)).getPath();
      }
      return serverDirectory;
   }

   protected String getServerBootTimeoutPropertyName()
   {
      return BOOT_TIMEOUT_PROPERTY_NAME;
   }

   protected String getDeploymentExceptionTransformerPropertyName()
   {
      return DEPLOYMENT_EXCEPTION_TRANSFORMER_PROPERTY_NAME;
   }

   protected DeploymentExceptionTransformer getDeploymentExceptionTransformer()
   {
      if (deploymentExceptionTransformer == null)
      {
         this.deploymentExceptionTransformer = getProperties().getInstanceValue(getDeploymentExceptionTransformerPropertyName(), DeploymentExceptionTransformer.class, false);
         if (deploymentExceptionTransformer == null)
         {
            this.deploymentExceptionTransformer = new DeploymentExceptionTransformer()
            {

               public DeploymentException transform(DeploymentException exception)
               {
                  return exception;
               }

            };
         }
      }
      return deploymentExceptionTransformer;
   }

   protected Long getServerBootTimeout()
   {
      if (bootTimeout == null)
      {
         this.bootTimeout = getProperties().getLongValue(getServerBootTimeoutPropertyName(), 240000, false);
      }
      return bootTimeout;
   }

   protected String getForceRestartPropertyName()
   {
      return FORCE_RESTART_PROPERTY_NAME;
   }

   protected Boolean getForceRestart()
   {
      if (forceRestart == null)
      {
         this.forceRestart = this.forceRestart = getProperties().getBooleanValue(getForceRestartPropertyName(), false, false);
      }
      return forceRestart;
   }

   protected String getHostPropertyName()
   {
      return HOST_PROPERTY_NAME;
   }

   protected String getHost()
   {
      if (host == null)
      {
         host = getProperties().getStringValue(getHostPropertyName(), "localhost", false);
      }
      return host;
   }

   protected String getPortPropertyName()
   {
      return PORT_PROPERTY_NAME;
   }

   protected String getPort()
   {
      if (port == null)
      {
         port = getProperties().getStringValue(getPortPropertyName(), "8080", false);
      }
      return port;
   }

   protected String getShutdownDelayPropertyName()
   {
      return SHUTDOWN_DELAY_PROPERTY_NAME;
   }

   protected Integer getShutdownDelay()
   {
      if (shutdownDelay == null)
      {
         this.shutdownDelay = getProperties().getIntValue(SHUTDOWN_DELAY_PROPERTY_NAME, 15000, false);
      }
      return shutdownDelay;
   }

   protected PropertiesManager getProperties()
   {
      return properties;
   } 

   protected String getJavaOpts()
   {
      if (javaOpts == null)
      {
         javaOpts = getProperties().getStringValue(JAVA_OPTS_PROPERTY_NAME, "", false);
         javaOpts = javaOpts + JAVA_OPTS;
      }
      return javaOpts;
   }

   protected String getHttpUrl()
   {
      if (httpUrl == null)
      {
         this.httpUrl = "http://" + getHost() + ":" + getPort() + "/";
      }
      return httpUrl;
   }

   protected String getExtraConfigurationDirPropertyName()
   {
      return EXTRA_CONFIGURATION_DIR_PROPERTY_NAME;
   }

   protected abstract String getLogName(); 

   protected String getExtraConfigurationDir()
   {
      if (extraConfigurationDir == null)
      {
         this.extraConfigurationDir = getProperties().getStringValue(EXTRA_CONFIGURATION_DIR_PROPERTY_NAME, null, false);
      }
      return extraConfigurationDir;
   }

}