/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.threads.metadata;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.jboss.aop.microcontainer.aspects.jmx.JMX;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.logging.Logger;
import org.jboss.threads.ArrayQueue;
import org.jboss.threads.DirectExecutor;
import org.jboss.threads.JBossExecutors;
import org.jboss.threads.JBossThreadFactory;
import org.jboss.threads.JBossThreadPoolExecutor;
import org.jboss.threads.OrderedExecutor;
import org.jboss.threads.QueueExecutor;
import org.jboss.threads.QueuelessExecutor;
import org.jboss.threads.management.BoundedQueueThreadPoolExecutorMBean;
import org.jboss.threads.management.BoundedThreadPoolExecutorMBean;
import org.jboss.threads.management.ThreadExecutorMBean;
import org.jboss.threads.management.ThreadPoolExecutorMBean;

import javax.management.ObjectName;

public final class ThreadsHelper {

    private ThreadsHelper() {
    }

    static final Map<String, String> UNIT_NICK_NAMES = stringMap(
        entry("S", "SECONDS"),
        entry("SEC", "SECONDS"),
        entry("SECOND", "SECONDS"),
        entry("SECONDS", "SECONDS"),
        entry("M", "MINUTES"),
        entry("MIN", "MINUTES"),
        entry("MINUTE", "MINUTES"),
        entry("MINUTES", "MINUTES"),
        entry("MS", "MILLISECONDS"),
        entry("MILLIS", "MILLISECONDS"),
        entry("MILLISECOND", "MILLISECONDS"),
        entry("MILLISECONDS", "MILLISECONDS"),
        entry("NS", "NANOSECONDS"),
        entry("NANOS", "NANOSECONDS"),
        entry("NANOSECOND", "NANOSECONDS"),
        entry("NANOSECONDS", "NANOSECONDS"),
        entry("H", "HOURS"),
        entry("HOUR", "HOURS"),
        entry("HOURS", "HOURS"),
        entry("D", "DAYS"),
        entry("DAY", "DAYS"),
        entry("DAYS", "DAYS"),
        entry("MON", "MONTHS"),
        entry("MONTH", "MONTHS"),
        entry("MONTHS", "MONTHS"),
        entry("W", "WEEKS"),
        entry("WEEK", "WEEKS"),
        entry("WEEKS", "WEEKS")
    );

    public static Runnable createTaskFilterTask(final List<Object> objects, final Runnable task) {
        return JBossExecutors.executorTask(createTaskFilter(objects), task);
    }

    public static DirectExecutor createTaskFilter(final List<Object> objects) {
        return createTaskFilterRecursive(objects.iterator());
    }

    static DirectExecutor createTaskFilterRecursive(final Iterator<Object> it) {
        if (! it.hasNext()) {
            return JBossExecutors.directExecutor();
        }
        final Object object = it.next();
        if (object instanceof DirectExecutor && ! it.hasNext()) {
            return (DirectExecutor) object;
        }
        switch ((FilterTypes) object) {
            case THREAD_NAME: {
                final String nextString = (String) it.next();
                return JBossExecutors.threadNameExecutor(createTaskFilterRecursive(it), nextString);
            }
            case THREAD_NAME_NOTATION: {
                final String nextString = (String) it.next();
                return JBossExecutors.threadNameNotateExecutor(createTaskFilterRecursive(it), nextString);
            }
            case LOG_EXCEPTIONS: {
                final Logger log = Logger.getLogger((String) it.next());
                final Logger.Level level = Logger.Level.valueOf((String) it.next());
                return JBossExecutors.exceptionLoggingExecutor(createTaskFilterRecursive(it), log, level);
            }
            case CLEAR_TLS: {
                return JBossExecutors.resettingExecutor(createTaskFilterRecursive(it));
            }
            case CLEAR_CONTEXT_CLASSLOADER: {
                return JBossExecutors.cleanupExecutor(createTaskFilterRecursive(it), JBossExecutors.contextClassLoaderResetter());
            }
            case INITIALIZER: {
                final Runnable r = (Runnable) it.next();
                return JBossExecutors.initializingExecutor(createTaskFilterRecursive(it), r);
            }
            case CLEANER: {
                final Runnable r = (Runnable) it.next();
                return JBossExecutors.cleanupExecutor(createTaskFilterRecursive(it), r);
            }
            default: {
                throw new IllegalStateException();
            }
        }
    }

    public static Thread createThread(final ThreadFactory threadFactory, final List<Object> taskFilterObjects, final Runnable task) {
        return threadFactory.newThread(createTaskFilterTask(taskFilterObjects, task));
    }

    public static ThreadFactory createThreadFactory(final ThreadGroup threadGroup, final Boolean daemon, final Integer priority, final String namePattern, final Thread.UncaughtExceptionHandler exceptionHandler, final Long stackSize, final List<Object> taskFilterObjects) {
        return JBossExecutors.wrappingThreadFactory(createTaskFilter(taskFilterObjects), new JBossThreadFactory(threadGroup, daemon, priority, namePattern, exceptionHandler, stackSize));
    }

    public static Executor createJBossThreadPoolExecutor(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, List<Object> taskFilterObjects) {
        return JBossExecutors.wrappingExecutor(createTaskFilter(taskFilterObjects), new JBossThreadPoolExecutor(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory));
    }

    public static Executor createThreadFactoryExecutor(final ThreadFactory factory, int maxThreads, boolean blocking, final List<Object> taskFilterObjects) {
        return JBossExecutors.threadFactoryExecutor(factory, maxThreads, blocking, createTaskFilter(taskFilterObjects));
    }

    public static QueueExecutor createQueueExecutor(final int corePoolSize, final int maxPoolSize, final long keepAliveTime, final TimeUnit keepAliveTimeUnit, final Queue<Runnable> queue, final ThreadFactory threadFactory, final boolean blocking, final Executor handoffExecutor, final List<Object> taskFilterObjects) {
        return new QueueExecutor(corePoolSize, maxPoolSize, keepAliveTime, keepAliveTimeUnit, queue, threadFactory, blocking, handoffExecutor, createTaskFilter(taskFilterObjects)); 
    }

    public static QueuelessExecutor createQueuelessExecutor(final ThreadFactory threadFactory, final List<Object> taskFilterObjects, final Executor handoffExecutor, final long idleTimeout) {
        return new QueuelessExecutor(threadFactory, createTaskFilter(taskFilterObjects), handoffExecutor, idleTimeout);
    }

    static int calculateScaledCount(ScaledCountMetaData poolSizeMetaData) {
        float count = poolSizeMetaData.getCount();
        float perCpu = poolSizeMetaData.getPerCpu();
        if (Float.isNaN(count) || Float.isInfinite(count) || count < 0.0f) {
            count = 0.0f;
        }
        if (Float.isNaN(perCpu) || Float.isInfinite(perCpu) || perCpu < 0.0f) {
            perCpu = 0.0f;
        }
        return Math.round(count + ((float) Runtime.getRuntime().availableProcessors()) * perCpu);
    }

    private static StringEntry entry(final String key, final String value) {
        return new StringEntry(key, value);
    }

    private static Map<String, String> stringMap(StringEntry... entries) {
        final HashMap<String, String> hashMap = new HashMap<String, String>(entries.length);
        for (Map.Entry<String, String> e : entries) {
            hashMap.put(e.getKey(), e.getValue());
        }
        return Collections.unmodifiableMap(hashMap);
    }

    static void addMetaData(final List<BeanMetaData> beanMetaDataList, final ThreadMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder;
        final RefMetaData threadFactoryRef = metaData.getThreadFactory();
        builder = BeanMetaDataBuilder.createBuilder(name, Thread.class.getName());
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createThread");
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createInject(threadFactoryRef.getName()));
            final List<ValueMetaData> objectList = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, objectList);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData)objectList);
            builder.addConstructorParameter(Runnable.class.getName(), builder.createInject(metaData.getTask().getName()));
        } else {
            builder.setFactory(builder.createInject(threadFactoryRef.getName()));
            builder.addConstructorParameter(Runnable.class.getName(), builder.createInject(metaData.getTask().getName()));
        }
        final Boolean daemon = metaData.isDaemon();
        if (daemon != null) {
            builder.addPropertyMetaData("daemon", daemon);
        }
        final Integer priority = metaData.getPriority();
        if (priority != null) {
            builder.addPropertyMetaData("priority", priority);
        }
        final RefMetaData exceptionHandler = metaData.getExceptionHandler();
        if (exceptionHandler != null) {
            builder.addPropertyMetaData("uncaughtExceptionHandler", builder.createInject(exceptionHandler.getName()));
        }
        builder.ignoreCreate();
        builder.setStop("interrupt");
        builder.ignoreDestroy();
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    static void addMetaData(final List<BeanMetaData> beanMetaDataList, final ThreadGroupMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, ThreadGroup.class.getName());
        final RefMetaData parentRef = metaData.getParentThreadGroup();
        if (parentRef != null) {
            builder.addConstructorParameter(ThreadGroup.class.getName(), builder.createInject(parentRef.getName()));
        }
        final String groupName = metaData.getGroupName();
        builder.addConstructorParameter(String.class.getName(), builder.createValue(groupName != null ? groupName : name));
        if (metaData.isDaemon() != null) {
            builder.addPropertyMetaData("daemon", builder.createValue(metaData.isDaemon()));
        }
        final Integer maxPriorityMeta = metaData.getMaxPriority();
        if (maxPriorityMeta != null) {
            builder.addPropertyMetaData("maxPriority", builder.createValue(maxPriorityMeta));
        }
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.setStop("interrupt");
        builder.ignoreDestroy();
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    static void addMetaData(final List<BeanMetaData> beanMetaDataList, final ThreadFactoryMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, JBossThreadFactory.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        final RefMetaData groupRef = metaData.getThreadGroup();
        builder.addConstructorParameter(ThreadGroup.class.getName(), groupRef == null ? builder.createNull() : builder.createInject(groupRef.getName()));
        final Boolean daemon = metaData.getDaemon();
        builder.addConstructorParameter(Boolean.class.getName(), daemon == null ? builder.createNull() : builder.createValue(daemon));
        final Integer priorityMeta = metaData.getPriority();
        final Integer actualPriorityMeta;
        if (priorityMeta != null) {
            actualPriorityMeta = priorityMeta;
        } else {
            actualPriorityMeta = null;
        }
        builder.addConstructorParameter(Integer.class.getName(), actualPriorityMeta == null ? builder.createNull() : builder.createValue(actualPriorityMeta));
        builder.addConstructorParameter(String.class.getName(), builder.createValue(metaData.getThreadNamePattern()));
        final RefMetaData exceptionHandler = metaData.getExceptionHandler();
        builder.addConstructorParameter(Thread.UncaughtExceptionHandler.class.getName(), exceptionHandler == null ? builder.createNull() : builder.createInject(exceptionHandler.getName()));
        builder.addConstructorParameter(Long.class.getName(), builder.createNull());
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createThreadFactory");
            final List<ValueMetaData> list = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, list);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData) list);
        }
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    static void addMetaData(final List<BeanMetaData> beanMetaDataList, final ThreadFactoryExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, Executor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        builder.setFactoryClass(JBossExecutors.class.getName());
        builder.setFactoryMethod("threadFactoryExecutor");
        final RefMetaData threadFactory = metaData.getThreadFactory();
        if (threadFactory != null) {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createInject(threadFactory.getName()));
        } else {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createValue(Executors.defaultThreadFactory()));
        }
        final ScaledCountMetaData maxThreads = metaData.getMaxThreads();
        if (maxThreads != null) {
            builder.addConstructorParameter(Integer.class.getName(), Integer.valueOf(calculateScaledCount(maxThreads)));
        } else {
            builder.addConstructorParameter(Integer.class.getName(), Integer.valueOf(Integer.MAX_VALUE));
        }
        builder.addConstructorParameter(Boolean.class.getName(), Boolean.valueOf(metaData.isBlocking()));
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createThreadFactoryExecutor");
            final List<ValueMetaData> list = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, list);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData) list);
        }
        builder.addAnnotation(new LiteralJMX(ThreadExecutorMBean.class, "jboss.threads:type=threadFactoryExecutor,name=" + ObjectName.quote(name), false));
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    static void addMetaData(final List<BeanMetaData> beanMetaDataList, final UnboundedQueueThreadPoolExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, Executor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.setStop("shutdown");
        builder.ignoreDestroy();
        final int coreThreads = calculateScaledCount(metaData.getMaxThreads());
        builder.addConstructorParameter("int", Integer.valueOf(coreThreads));
        builder.addConstructorParameter("int", Integer.valueOf(coreThreads));
        final TimeMetaData keepaliveTime = metaData.getKeepaliveTime();
        if (keepaliveTime != null) {
            builder.addConstructorParameter("long", Long.valueOf(keepaliveTime.getTime()));
            builder.addConstructorParameter(TimeUnit.class.getName(), builder.createValue(TimeUnit.valueOf(UNIT_NICK_NAMES.get(keepaliveTime.getUnit().toUpperCase()))));
            builder.addPropertyMetaData("allowCoreThreadTimeout", Boolean.TRUE);
        } else {
            builder.addConstructorParameter("long", Long.valueOf(Long.MAX_VALUE));
            builder.addConstructorParameter(TimeUnit.class.getName(), builder.createValue(TimeUnit.SECONDS));
        }
        builder.addConstructorParameter(Queue.class.getName(), builder.createValue(new LinkedBlockingQueue<Runnable>()));
        final RefMetaData threadFactory = metaData.getThreadFactory();
        if (threadFactory != null) {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createInject(threadFactory.getName()));
        } else {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createValue(Executors.defaultThreadFactory()));
        }
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createJBossThreadPoolExecutor");
            final List<ValueMetaData> list = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, list);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData) list);
        }
        builder.addAnnotation(new LiteralJMX(ThreadPoolExecutorMBean.class, "jboss.threads:type=unboundedQueueThreadPool,name=" + name, false));
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    public static void addMetaData(final List<BeanMetaData> beanMetaDataList, final BoundedQueueThreadPoolExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, QueueExecutor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.setStop("shutdown");
        builder.ignoreDestroy();
        final ScaledCountMetaData coreThreads = metaData.getCoreThreads();
        final ScaledCountMetaData maxThreads = metaData.getMaxThreads();
        if (coreThreads != null) {
            builder.addConstructorParameter("int", Integer.valueOf(calculateScaledCount(coreThreads)));
        } else {
            builder.addConstructorParameter("int", Integer.valueOf(calculateScaledCount(maxThreads)));
        }
        builder.addConstructorParameter("int", Integer.valueOf(calculateScaledCount(maxThreads)));
        final TimeMetaData keepaliveTime = metaData.getKeepaliveTime();
        if (keepaliveTime != null) {
            builder.addConstructorParameter("long", Long.valueOf(keepaliveTime.getTime()));
            builder.addConstructorParameter(TimeUnit.class.getName(), builder.createValue(TimeUnit.valueOf(UNIT_NICK_NAMES.get(keepaliveTime.getUnit().toUpperCase()))));
        } else {
            builder.addConstructorParameter("long", Long.valueOf(Long.MAX_VALUE));
            builder.addConstructorParameter(TimeUnit.class.getName(), builder.createValue(TimeUnit.SECONDS));
        }
        builder.addConstructorParameter(Queue.class.getName(), builder.createValue(new ArrayQueue<Runnable>(calculateScaledCount(metaData.getQueueLength()))));
        final RefMetaData threadFactory = metaData.getThreadFactory();
        if (threadFactory != null) {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createInject(threadFactory.getName()));
        }
        builder.addConstructorParameter("boolean", Boolean.valueOf(metaData.isBlocking()));
        final RefMetaData handoffExecutor = metaData.getHandoffExecutor();
        if (handoffExecutor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(handoffExecutor.getName()));
        } else {
            builder.addConstructorParameter(Executor.class.getName(), builder.createNull());
        }
        if (metaData.isAllowCoreTimeout()) {
            builder.addPropertyMetaData("allowCoreTimeout", Boolean.TRUE);
        }
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createQueueExecutor");
            final List<ValueMetaData> list = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, list);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData) list);
        }
        builder.addAnnotation(new LiteralJMX(BoundedQueueThreadPoolExecutorMBean.class, "jboss.threads:type=boundedQueueThreadPool,name=" + name, false));
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    public static void addMetaData(final List<BeanMetaData> beanMetaDataList, final QueuelessThreadPoolExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, QueuelessExecutor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.setStop("shutdown");
        builder.ignoreDestroy();
        final RefMetaData threadFactory = metaData.getThreadFactory();
        if (threadFactory != null) {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createInject(threadFactory.getName()));
        } else {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createValue(Executors.defaultThreadFactory()));
        }
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createQueuelessExecutor");
            final List<ValueMetaData> list = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, list);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData) list);
        } else {
            builder.addConstructorParameter(DirectExecutor.class.getName(), builder.createValue(JBossExecutors.directExecutor()));
        }
        final RefMetaData handoffExecutor = metaData.getHandoffExecutor();
        if (handoffExecutor != null) {
            builder.addConstructorParameter(Executor.class.getName(), builder.createInject(handoffExecutor.getName()));
        } else {
            builder.addConstructorParameter(Executor.class.getName(), builder.createNull());
        }
        final TimeMetaData keepaliveTime = metaData.getKeepaliveTime();
        if (keepaliveTime != null) {
            final TimeUnit unit = TimeUnit.valueOf(UNIT_NICK_NAMES.get(keepaliveTime.getUnit().toUpperCase()));
            builder.addConstructorParameter("long", builder.createValue(Long.valueOf(unit.toMillis(keepaliveTime.getTime()))));
        } else {
            builder.addConstructorParameter("long", builder.createValue(Long.valueOf(Long.MAX_VALUE)));
        }
        final ScaledCountMetaData maxThreads = metaData.getMaxThreads();
        if (maxThreads != null) {
            builder.addPropertyMetaData("maxThreads", Integer.valueOf(calculateScaledCount(maxThreads)));
        } else {
            builder.addPropertyMetaData("maxThreads", Integer.valueOf(Integer.MAX_VALUE));
        }
        if (metaData.isBlocking()) {
            builder.addPropertyMetaData("blocking", Boolean.TRUE);
        }
        builder.addAnnotation(new LiteralJMX(BoundedThreadPoolExecutorMBean.class, "jboss.threads:type=queuelessThreadPool,name=" + name, false));
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    public static void addMetaData(final List<BeanMetaData> beanMetaDataList, final ScheduledThreadPoolExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, QueuelessExecutor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.setStop("shutdown");
        builder.ignoreDestroy();
        final ScaledCountMetaData coreThreads = metaData.getMaxThreads();
        builder.addConstructorParameter("int", coreThreads == null ? Integer.valueOf(1) : Integer.valueOf(calculateScaledCount(coreThreads)));
        final RefMetaData threadFactory = metaData.getThreadFactory();
        if (threadFactory != null) {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createInject(threadFactory.getName()));
        } else {
            builder.addConstructorParameter(ThreadFactory.class.getName(), builder.createValue(Executors.defaultThreadFactory()));
        }
        final TimeMetaData keepaliveTime = metaData.getKeepaliveTime();
        if (keepaliveTime != null) {
            final TimeUnit unit = TimeUnit.valueOf(UNIT_NICK_NAMES.get(keepaliveTime.getUnit().toUpperCase()));
            builder.addPropertyMetaData("keepAliveTime", builder.createValue(Long.valueOf(unit.toMillis(keepaliveTime.getTime()))));
        }
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            throw new UnsupportedOperationException("task-filter not yet supported on scheduled-thread-pool-executor");
        }
        builder.addAnnotation(new LiteralJMX(ThreadPoolExecutorMBean.class, "jboss.threads:type=scheduledThreadPool,name=" + name, false));
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    public static void addMetaData(final List<BeanMetaData> beanMetaDataList, final UnboundedOrderedExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, OrderedExecutor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        builder.addConstructorParameter(Executor.class.getName(), builder.createInject(metaData.getParentExecutor().getName()));
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    public static void addMetaData(final List<BeanMetaData> beanMetaDataList, final BoundedOrderedExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, OrderedExecutor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        builder.addConstructorParameter(Executor.class.getName(), builder.createInject(metaData.getParentExecutor().getName()));
        builder.addConstructorParameter("int", Integer.valueOf(calculateScaledCount(metaData.getQueueLength())));
        builder.addConstructorParameter("boolean", Boolean.valueOf(metaData.isBlocking()));
        final RefMetaData handoffExecutor = metaData.getHandoffExecutor();
        builder.addConstructorParameter(Executor.class.getName(), handoffExecutor != null ? builder.createInject(handoffExecutor.getName()) : builder.createNull());
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    public static void addMetaData(final List<BeanMetaData> beanMetaDataList, final DirectExecutorMetaData metaData) {
        final String name = metaData.getName();
        final BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, DirectExecutor.class.getName());
        builder.ignoreCreate();
        builder.ignoreStart();
        builder.ignoreStop();
        builder.ignoreDestroy();
        final TaskFilterMetaData taskFilter = metaData.getTaskFilter();
        if (taskFilter != null) {
            builder.setFactoryClass(ThreadsHelper.class.getName());
            builder.setFactoryMethod("createTaskFilter");
            final List<ValueMetaData> list = builder.createList(ArrayList.class.getName(), Object.class.getName());
            for (AbstractTaskFilter filter : taskFilter.getTaskFilters()) {
                filter.addTo(builder, list);
            }
            builder.addConstructorParameter(List.class.getName(), (ValueMetaData) list);
        } else {
            builder.setFactoryClass(JBossExecutors.class.getName());
            builder.setFactoryMethod("directExecutor");
        }
        beanMetaDataList.add(builder.getBeanMetaData());
    }

    enum FilterTypes {
        THREAD_NAME,
        THREAD_NAME_NOTATION,
        LOG_EXCEPTIONS,
        CLEAR_TLS,
        CLEAR_CONTEXT_CLASSLOADER,
        INITIALIZER,
        CLEANER,
    }

    private static final class StringEntry implements Map.Entry<String, String> {
        private final String key;
        private final String value;

        private StringEntry(final String key, final String value) {
            this.key = key;
            this.value = value;
        }

        public String getKey() {
            return key;
        }

        public String getValue() {
            return value;
        }

        public String setValue(final String value) {
            throw new UnsupportedOperationException();
        }
    }

    private static class LiteralJMX implements JMX {
        private final Class<?> exposedInterface;
        private final String name;
        private final boolean registerDirectly;

        private LiteralJMX(final Class<?> exposedInterface, final String name, final boolean registerDirectly) {
            this.exposedInterface = exposedInterface;
            this.name = name;
            this.registerDirectly = registerDirectly;
        }

        public Class<?> exposedInterface() {
            return exposedInterface;
        }

        public String name() {
            return name;
        }

        public boolean registerDirectly() {
            return registerDirectly;
        }

        public Class<? extends Annotation> annotationType() {
            return JMX.class;
        }
    }
}
