/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.tmpdpl.api.deployable;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jboss.shrinkwrap.api.Archive;

/**
 * VfsVdfDeployableFactory
 * 
 * Factory to create {@link Deployable} instances from 
 * {@link Archive}s.  This removes the API
 * dependency upon internals for the client view and additionally
 * acts as a convenience mechanism.
 *
 * @author <a href="mailto:andrew.rubinger@jboss.org">ALR</a>
 * @version $Revision: $
 */
public class VfsVdfDeployableFactory
{

   //-------------------------------------------------------------------------------------||
   // Class Members ----------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Logger
    */
   private static final Logger log = Logger.getLogger(VfsVdfDeployableFactory.class.getName());

   /**
    * FQN of implementation Class used in creating new Deployables 
    */
   private static final String CLASS_NAME_VFS_VDF_DEPLOYABLE = "org.jboss.tmpdpl.impl.vdf.VfsVdfDeployableImpl";

   /**
    * FQN of the archive type actually supported by {@link VfsVdfDeployableFactory#createDeployable(VirtualArchive)}
    */
   private static final String CLASS_NAME_ARCHIVE_TYPE = "org.jboss.shrinkwrap.api.Archive";

   /**
    * Name of the method used for creation
    */
   private static final String METHOD_NAME_CREATE = "create";

   /**
    * Method used in creating new {@link Deployable} instances
    */
   private static Method method;

   //-------------------------------------------------------------------------------------||
   // Constructor ------------------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Internal Constructor to prohibit external
    * instantiation
    */
   private VfsVdfDeployableFactory()
   {
   }

   //-------------------------------------------------------------------------------------||
   // Functional Methods -----------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Creates a {@link Deployable} from the specified archive
    * 
    * @param archive
    * @throws IllegalArgumentException If the archive is not specified 
    */
   public static Deployable createDeployable(final Archive<?> archive) throws IllegalArgumentException
   {
      // Precondition check
      if (archive == null)
      {
         throw new IllegalArgumentException("Archive must be specified");
      }

      final Object obj;
      try
      {
         obj = getMethod().invoke(null, archive);
         if (log.isLoggable(Level.FINER))
         {
            log.log(Level.FINER, "Created: " + obj);
         }
      }
      catch (final IllegalAccessException iae)
      {
         throw new RuntimeException("Could not invoke static method: " + method, iae);
      }
      catch (final InvocationTargetException ite)
      {
         throw new RuntimeException("Could not invoke static method: " + method, ite);
      }

      // Cast 
      final Deployable deployable;
      try
      {
         deployable = Deployable.class.cast(obj);
      }
      catch (final ClassCastException cce)
      {
         throw new RuntimeException("New instance should be of type " + Deployable.class.getName(), cce);
      }

      // Return
      return deployable;
   }

   //-------------------------------------------------------------------------------------||
   // Internal Helper Methods ------------------------------------------------------------||
   //-------------------------------------------------------------------------------------||

   /**
    * Obtains the method used for creating new Deployable instances.
    * Uses a cached copy unless not yet initialized.
    */
   private synchronized static Method getMethod()
   {
      // If we haven't yet cached the ctor
      if (method == null)
      {
         // Get the static method
         final Class<?> clazz = getClass(CLASS_NAME_VFS_VDF_DEPLOYABLE);
         final Class<?> paramType = getClass(CLASS_NAME_ARCHIVE_TYPE);
         try
         {
            method = clazz.getDeclaredMethod(METHOD_NAME_CREATE, paramType);
         }
         catch (final NoSuchMethodException nsme)
         {
            throw new RuntimeException("Could not obtain method \"" + METHOD_NAME_CREATE + " from " + clazz.getName(),
                  nsme);
         }

      }

      // Return
      return method;
   }

   /**
    * Obtains the class with the specified name from the TCCL
    *  
    * @param className
    * @return
    */
   private static Class<?> getClass(final String className)
   {
      final ClassLoader cl = SecurityActions.getThreadContextClassLoader();
      try
      {
         return Class.forName(className, false, cl);
      }
      catch (ClassNotFoundException cnfe)
      {
         throw new RuntimeException("Could not find implementation class \"" + className + "\" in " + cl, cnfe);
      }
   }

}
