/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.remote.driver;

import org.jboss.beans.metadata.plugins.AbstractBeanMetaData;
import org.jboss.beans.metadata.plugins.AbstractConstructorMetaData;
import org.jboss.beans.metadata.plugins.AbstractValueMetaData;
import org.jboss.beans.metadata.api.annotations.Inject;
import org.jboss.kernel.plugins.bootstrap.basic.BasicBootstrap;
import org.jboss.kernel.plugins.deployment.xml.BeanXMLDeployer;
import org.jboss.kernel.spi.deployment.KernelDeployment;
import org.jboss.portal.test.framework.deployment.Deployer;
import org.jboss.portal.test.framework.server.Node;
import org.jboss.portal.test.framework.server.NodeId;
import org.jboss.portal.test.framework.server.NodeManager;
import org.jboss.unit.Failure;
import org.jboss.unit.TestId;
import org.jboss.unit.driver.DriverCommand;
import org.jboss.unit.driver.DriverContext;
import org.jboss.unit.driver.DriverException;
import org.jboss.unit.driver.DriverResponse;
import org.jboss.unit.driver.TestDriver;
import org.jboss.unit.driver.command.StartTestCommand;
import org.jboss.unit.driver.response.FailureResponse;
import org.jboss.unit.info.TestInfo;
import org.jboss.unit.info.impl.SimpleTestInfo;
import org.jboss.logging.Logger;

import java.net.URL;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class RemoteTestDriverClient extends BasicBootstrap implements TestDriver
{

   /** . */
   private static Logger log = org.jboss.logging.Logger.getLogger(RemoteTestDriverClient.class);

   // Config

   /** Need to make initial node id configurable somehow. */
//   private String initialNodeId;

   // Services

   /** . */
   private SimpleTestInfo info;

   /** . */
   private DriverContext context;

   /** . */
   private NodeManager nodeManager;

   /** . */
   private ProtocolHandler protocolHandler;

   /** . */
   private String archivePath;

   /** . */
   private String serverName;

   /** . */
   private Deployer deployer;

//   public String getInitialNodeId()
//   {
//      return initialNodeId;
//   }
//
//   public void setInitialNodeId(String initialNodeId)
//   {
//      this.initialNodeId = initialNodeId;
//   }

   public String getArchivePath()
   {
      return archivePath;
   }

   public String getServerName()
   {
      return serverName;
   }

   public NodeManager getNodeManager()
   {
      return nodeManager;
   }

   @Inject(bean="NodeManager")
   public void setNodeManager(NodeManager nodeManager)
   {
      this.nodeManager = nodeManager;
   }

   public ProtocolHandler getProtocolHandler()
   {
      return protocolHandler;
   }

   @Inject(bean="DeployerHandler")
   public void setProtocolHandler(ProtocolHandler protocolHandler)
   {
      this.protocolHandler = protocolHandler;
   }

   public Deployer getDeployer()
   {
      return deployer;
   }

   @Inject(bean="Deployer")
   public void setDeployer(Deployer deployer)
   {
      this.deployer = deployer;
   }

   public RemoteTestDriver getServer(Node node)
   {
      return (RemoteTestDriver)node.lookupService(RemoteTestDriver.SERVICE_ID);
   }

   private KernelDeployment deployment;
   private BeanXMLDeployer beanDeployer;

   public void initDriver(DriverContext context) throws DriverException
   {
      this.context = context;

      // Get this property
      this.archivePath = context.getProperty("archivePath");
      this.serverName = context.getProperty("serverName");

      //
      log.debug("Initializing remote test driving client");
      log.debug("Archive path = " + archivePath);
      log.debug("Server name = " + serverName);

      //
      log.debug("Booting kernel");
      run();

      // Install our self first
      try
      {
         log.debug("Installing TestDriver bean");
         AbstractBeanMetaData bmd = new AbstractBeanMetaData("TestDriver", getClass().getName());
         AbstractConstructorMetaData cmd = new AbstractConstructorMetaData();
         bmd.setConstructor(cmd);
         TestCaseFactory factory = new TestCaseFactory(this);
         AbstractValueMetaData vmd = new AbstractValueMetaData(factory);
         cmd.setFactory(vmd);
         cmd.setFactoryMethod("getTestCase");
         getKernel().getController().install(bmd);
      }
      catch (Throwable throwable)
      {
         throw new DriverException("Could not install the TestDriver bean", throwable);
      }

      // Now deploy beans
      try
      {
         String location = "/org/jboss/unit/remote/driver/jboss-beans.xml";
         URL url = getClass().getResource(location);
         if (url == null)
         {
            log.debug("The file jboss-beans.xml supposed to be located under " + location + " could not be found");
         }

         //
         log.debug("Now deploying the beans with url " + url);
         beanDeployer = new BeanXMLDeployer(getKernel());
         deployment = beanDeployer.deploy(url);
         beanDeployer.validate();

         // Perform the remote deployment if necessary
         String archiveId = context.getProperty("archiveId");
         if (archiveId != null)
         {
            log.debug("Going to deploy " + archiveId);
            Node node = nodeManager.getNode(NodeId.DEFAULT);
            deployer.deploy(archivePath, archiveId, node);
         }
         else
         {
            log.debug("No archive id was provided");
         }
      }
      catch (Throwable throwable)
      {
         throw new DriverException("Could not deploy the beans", throwable);
      }

      // Always retrieve meta data from the default node
      Node node = nodeManager.getNode(NodeId.DEFAULT);

      //
      TestDriver server = getServer(node);

      //
      server.initDriver(context);

      //
      info = SimpleTestInfo.create(server.getInfo());
   }

   public void destroyDriver()
   {
      try
      {
         // Always retrieve meta data from the default node
         Node node = nodeManager.getNode(NodeId.DEFAULT);

         //
         TestDriver server = getServer(node);

         //
         server.destroyDriver();
      }
      catch (Exception ignore)
      {
         ignore.printStackTrace();
      }

      //
      try
      {
         beanDeployer.undeploy(deployment);
      }
      catch (Throwable ignore)
      {
         ignore.printStackTrace();
      }

      //
      this.context = null;
   }

   public TestInfo getInfo()
   {
      return info;
   }

   public DriverResponse invoke(TestId testId, DriverCommand command)
   {
      if (command instanceof StartTestCommand)
      {
         try
         {
            Node node = nodeManager.getNode(NodeId.DEFAULT);

            // Create conversation
            TestConversation conversation = createConversation(testId, node);

            //
            return conversation.handleCommand(command);
         }
         catch (Exception e)
         {
            return new FailureResponse(Failure.createFailure(e));
         }
      }
      else
      {
         return new FailureResponse(Failure.createErrorFailure("Command " + command + " not accepted"));
      }
   }

   protected TestConversation createConversation(TestId testId, Node node)
   {
      return new TestConversation(protocolHandler, this, testId, node);
   }

   public Node getNode(int port)
   {
      NodeId nodeId = null;
      switch (port)
      {
         case 8080:
            nodeId = NodeId.DEFAULT;
            break;
         case 8180:
            nodeId = NodeId.PORTS_01;
            break;
         case 8280:
            nodeId = NodeId.PORTS_02;
            break;
      }
      return nodeManager.getNode(nodeId);
   }

   public int getPort(Node node)
   {
      NodeId nodeId = node.getId();
      if (NodeId.DEFAULT.equals(nodeId))
      {
         return 8080;
      }
      if (NodeId.PORTS_01.equals(nodeId))
      {
         return 8180;
      }
      if (NodeId.PORTS_02.equals(nodeId))
      {
         return 8280;
      }
      return -1;
   }
}
