/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.remote.driver.handler.http;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.SimpleHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HeaderElement;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.ByteArrayRequestEntity;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.jboss.logging.Logger;
import org.jboss.unit.remote.http.HttpRequest;
import org.jboss.unit.remote.http.HttpHeaders;
import org.jboss.unit.remote.http.HttpHeader;
import org.jboss.portal.test.framework.server.Node;
import org.jboss.unit.remote.driver.handler.http.command.DoMethodCommand;
import org.jboss.unit.remote.driver.handler.http.command.DoPostCommand;
import org.jboss.unit.remote.driver.handler.http.command.DoGetCommand;
import org.jboss.unit.remote.driver.TestConversation;
import org.jboss.unit.remote.driver.RemoteDriverCommandContext;
import org.jboss.unit.remote.driver.RemoteDriverResponseContext;
import org.jboss.unit.remote.ResponseContext;
import org.jboss.unit.driver.DriverCommand;
import org.jboss.unit.driver.DriverResponse;
import org.jboss.unit.driver.response.FailureResponse;
import org.jboss.unit.Failure;

import java.net.URI;
import java.net.URL;
import java.net.URISyntaxException;
import java.net.MalformedURLException;
import java.util.Collection;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class HTTPConversation
{

   /** . */
   private final org.jboss.logging.Logger log = Logger.getLogger(getClass());

   /** . */
   private final HttpClient client;

   /** . */
   private final TestConversation conversation;

   public HTTPConversation(TestConversation conversation)
   {
      HttpClient client = new HttpClient(new SimpleHttpConnectionManager());
      client.getParams().setParameter(HttpMethodParams.RETRY_HANDLER, null);
      client.getState().setCredentials(AuthScope.ANY, new UsernamePasswordCredentials("test", "test"));

      //
      this.conversation = conversation;
      this.client = client;
   }

   public RemoteDriverResponseContext invoke(RemoteDriverCommandContext commandContext) throws Exception
   {
      DriverCommand command = commandContext.getCommand();

      //
      if (command instanceof DoMethodCommand)
      {
         DoMethodCommand method = (DoMethodCommand)command;
         URI uri = method.getURI();
         URL url = getURL(uri);
         if (command instanceof DoPostCommand)
         {
            DoPostCommand doPostCmd = (DoPostCommand)command;
            PostMethod post = null;
            try
            {
               post = new PostMethod(url.toString());
               post.setFollowRedirects(false);
               HttpRequest.Body body = doPostCmd.getBody();
               if (doPostCmd.getContentType() != null)
               {
                  post.addRequestHeader("Content-Type", doPostCmd.getContentType());
               }
               if (body instanceof HttpRequest.Raw)
               {
                  HttpRequest.Raw rb = (HttpRequest.Raw)body;
                  ByteArrayRequestEntity entity = new ByteArrayRequestEntity(rb.getBytes());
                  post.setRequestEntity(entity);
               }
               else if (body instanceof HttpRequest.Form)
               {
                  HttpRequest.Form fb = (HttpRequest.Form)body;
                  Collection<NameValuePair> tmp = new ArrayList<NameValuePair>();
                  for (Object o : fb.getParameterNames())
                  {
                     String name = (String)o;
                     String[] values = fb.getParameterValues(name);
                     for (String value : values)
                     {
                        NameValuePair nvp = new NameValuePair(name, value);
                        tmp.add(nvp);
                     }
                  }
                  NameValuePair[] nvps = tmp.toArray(new NameValuePair[tmp.size()]);
                  post.setRequestBody(nvps);
               }
               executeHTTPMethod(commandContext, post);
               return decodeHTTPResponse(commandContext, post);
            }
            finally
            {
               if (post != null)
               {
                  post.releaseConnection();
               }
            }
         }
         else
         {
            DoGetCommand doGetCmd = (DoGetCommand)command;
            GetMethod get = null;
            try
            {
               get = new GetMethod(url.toString());
               HttpHeaders headers = doGetCmd.getHeaders();
               for (Iterator i = headers.headers();i.hasNext();)
               {
                  HttpHeader header = (HttpHeader)i.next();
                  Header _header = new Header(header.getName(), header.getValue());
                  get.addRequestHeader(_header);
               }
               get.setFollowRedirects(false);
               executeHTTPMethod(commandContext, get);
               return decodeHTTPResponse(commandContext, get);
            }
            finally
            {
               if (get != null)
               {
                  get.releaseConnection();
               }
            }
         }
      }
      else
      {
         return commandContext.createResponseContext(new FailureResponse(Failure.createErrorFailure("Unexpected response")));
      }
   }

   private URL getURL(URI uri) throws URISyntaxException, MalformedURLException
   {
      if (!uri.isAbsolute())
      {
         Node node = conversation.getNode();
         int port = conversation.getDriver().getPort(node);
         uri = new URI("http://localhost:" + port).resolve(uri);
      }
      return uri.toURL();
   }

   private RemoteDriverResponseContext decodeHTTPResponse(RemoteDriverCommandContext commandContext, HttpMethod httpMethod) throws Exception
   {
      DriverResponse response = null;

      //
      ResponseContext ctx = conversation.popContext();
      if (ctx != null)
      {
         response = ctx.getResponse();
      }

      //
      int status = httpMethod.getStatusCode();
      switch (status)
      {
         case 200:
            // HTTP response is ok
            if (response == null)
            {
               response = new FailureResponse(Failure.createErrorFailure("No result for test " + conversation.getTestId() + " in the response"));
            }

            //
            log.info("# Received '200' code");
            break;
         case 302:
            // Send redirect
            if (response == null)
            {
               // Satisfy the 302 code
               Header locationHeader = httpMethod.getResponseHeader("location");
               if (locationHeader != null)
               {
                  String redirectLocation = locationHeader.getValue();
                  log.info("# Received '302' code --> " + redirectLocation);
                  DoGetCommand cmd = new DoGetCommand(new URI(redirectLocation));

                  // We should somehow stuff the response in the next payload
                  // but it's not yet proven it's usefull

                  // For now we don't add any contextual payload as
                  // 302 is some kind of implicit redirect response
                  return invoke(new RemoteDriverCommandContext(commandContext.getResponseContext(), cmd));
               }
               else
               {
                  // The response is invalid
                  response = new FailureResponse(Failure.createErrorFailure("302 Code with corrupted data"));
               }
            }
            else
            {
               // If any result has been setup during the action it overrides the 302 code
               log.info("# Received Result object which overrides the 302");
            }
            break;
         case 500:
            log.info("# Received '500' code");
            response = new FailureResponse(Failure.createErrorFailure("Received '500' code at " + httpMethod.getURI()));
            break;
         case 404:
            log.info("# Received '404' code");
            response = new FailureResponse(Failure.createErrorFailure("Received '404' code at " + httpMethod.getURI()));
            break;
         default:
            response = new FailureResponse(Failure.createErrorFailure("Unexpected http code " + status + " at " + httpMethod.getURI()));
            break;
      }

      //
      RemoteDriverResponseContext responseCtx = commandContext.createResponseContext(response);

      //
      byte[] body = httpMethod.getResponseBody();
      responseCtx.setPayload("http.response.body", body);

      //
      HttpHeaders _headers = new HttpHeaders();
      Header[] headers = httpMethod.getResponseHeaders();
      for (Header header : headers)
      {
         HttpHeader _header = _headers.addHeader(header.getName());
         HeaderElement[] elts = header.getElements();
         if (elts != null)
         {
            for (HeaderElement elt : elts)
            {
               HttpHeader.Element _elt = _header.addElement(elt.getName(), elt.getValue());
               NameValuePair[] params = elt.getParameters();
               if (params != null)
               {
                  for (NameValuePair param : params)
                  {
                     _elt.addParam(param.getName(), param.getValue());
                  }
               }
            }
         }
      }
      responseCtx.setPayload("http.response.headers", _headers);

      //
      return responseCtx;
   }

   private int executeHTTPMethod(RemoteDriverCommandContext commandContext, HttpMethod method) throws Exception
   {
      int port = method.getURI().getPort();

      // Update to the next node
      Node node = conversation.getDriver().getNode(port);
      if (node == null)
      {
         throw new IllegalArgumentException("Wrong port " + method.getURI().getPort() + " in URI " + method.getURI());
      }
      conversation.setNode(node);

      // Push context to the node
      conversation.pushContext(commandContext);

      //
      log.info("# Invoking test case over http " + method.getURI());
      int status = client.executeMethod(method);

      // Force to read the response body before we close the connection
      // otherwise the content will be lost
      method.getResponseBody();

      //
      return status;
   }
}
