/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.unit.tooling.maven2;

import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.project.MavenProject;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.Java;
import org.apache.tools.ant.types.Path;
import org.jboss.unit.tooling.JavaBuilder;
import org.jboss.unit.tooling.TaskExecutingThread;
import org.jboss.unit.tooling.TestFailedException;
import org.jboss.unit.tooling.maven2.configuration.CasesDescription;
import org.jboss.unit.tooling.maven2.configuration.Parameter;
import org.jboss.unit.tooling.maven2.configuration.Reports;
import org.jboss.unit.tooling.maven2.configuration.Testsuite;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.Arrays;
import java.util.Collection;

/**
 * @author <a href="mailto:boleslaw dot dawidowicz at redhat anotherdot com">Boleslaw Dawidowicz</a>
 * @version : 0.1 $
 */
public class TestSuiteExecutor
{
   private final MavenProject project;

   private final Testsuite testsuite;

   private final UnitMojo unitMojo;

   private TestSuiteExecutor()
   {
      project = null;
      testsuite = null;
      unitMojo = null;
   }

   public TestSuiteExecutor(MavenProject project, Testsuite testsuite, UnitMojo unitMojo)
   {
      this.project = project;
      this.testsuite = testsuite;
      this.unitMojo = unitMojo;
   }

   public void execute() throws Exception
   {

      // Prepare all settings for java invocation

      Java javaTask = createJavaTask();

      System.out.println("Invoking java: " + javaTask.getCommandLine().toString());


      // This is really stupid workaround to the fact that maven eats whole output of invoked java task.
      // Should be investigated and fixed in more proper way ...

      //javaTask.setFork(true);

      File log = null;


      javaTask.setAppend(true);
      javaTask.setLogError(true);



      try
      {
         log = File.createTempFile("jboss-unit-" + System.currentTimeMillis(), "log");

         javaTask.setOutput(log);
         javaTask.setError(log);


         // Execution of jboss-unit

         TaskExecutingThread executor = new TaskExecutingThread(javaTask);
         executor.run();



      }
      catch (Exception e)
      {
         throw new TestFailedException("Testsuite executed with test failures");
      }
      finally
      {
         // Print the collected output

         BufferedReader br = new BufferedReader(new FileReader(log));
         String line = null;
         while (( line = br.readLine()) != null){
            System.out.println(line);
         }

         // Cleanup
         log.delete();
      }
   }

   /**
    * Creates ant Java object with all proper settings
    * @return
    */
   private Java createJavaTask()
   {
      Java javaTask = null;
      try
      {

         JavaBuilder javaBuilder = new JavaBuilder();

         // Apply all settings for Java object creation
         addArguments(javaBuilder);

         // Update settings using environment variables
         javaBuilder.applyVariables(System.getProperties());

         // Create Java object
         javaTask = javaBuilder.buildJava();


         // Apply proper classpath taken from maven project
         Path path = javaTask.createClasspath();

         try
         {
            for (String element : (Collection<String>)project.getTestClasspathElements())
            {
               path.createPathElement().setLocation(new File(element));
            }
         }
         catch (DependencyResolutionRequiredException e)
         {
            e.printStackTrace();
         }

      }
      catch (BuildException e)
      {
         throw new IllegalStateException("Failed to create <java> ant task: ", e);
      }

      return javaTask;
   }

   private void addArguments(JavaBuilder javaBuilder)
   {
      javaBuilder.setConfig(testsuite.getConfig());

      Reports reports = testsuite.getReports();

      if (reports != null && reports.getXml() != null)
      {
         javaBuilder.setXmlReportDir(reports.getXml());
      }

      if (reports != null && reports.getHtml() != null)
      {
         javaBuilder.setHtmlReportDir(reports.getHtml());
      }

      // JPDA properties

      
      // JPDA
      if (unitMojo.isJpda())
      {
         javaBuilder.setJPDA(true);
      }

      // JPDA port
      if (unitMojo.getJpdaPort() != null)
      {
         javaBuilder.setJPDAport(Integer.decode(unitMojo.getJpdaPort()));
      }

      // JPDA suspend
      javaBuilder.setSuspend(Boolean.parseBoolean(unitMojo.getJpdaSuspend()));

      // JUL

      javaBuilder.setUseJUL(unitMojo.isUseJUL());

      // Assertions

      javaBuilder.setAssertions(unitMojo.isAssertions());


      // Fail on error


      javaBuilder.setFailOnError(unitMojo.isFailOnError());


      // Fork
      javaBuilder.setFork(unitMojo.isFork());
      

      // Check if IDs to execute are not forced from the command line

      for (CasesDescription include : testsuite.getIncludes())
      {

         if (include.getIds() != null)
         {
            for (String id : include.getIds())
            {
               javaBuilder.addIncludeID(id);
            }

         }
         if (include.getKeywords() != null)
         {
            for (String keyword : include.getKeywords())
            {
               javaBuilder.addIncludeKeyword(keyword);
            }
         }
      }

      for (CasesDescription exclude : testsuite.getExcludes())
      {

         if (exclude.getIds() != null)
         {
            for (String id : exclude.getIds())
            {
               javaBuilder.addExcludeID(id);
            }

         }
         if (exclude.getKeywords() != null)
         {
            for (String keyword : exclude.getKeywords())
            {
               javaBuilder.addExcludeKeyword(keyword);
            }
         }
      }


      // Properties
      if (testsuite.getProperties().size() > 0)
      {

         for (Object propertyName : testsuite.getProperties().keySet())
         {
            javaBuilder.addProperty(propertyName.toString(),testsuite.getProperties().get(propertyName).toString());
         }

         javaBuilder.updateFQNProperties(System.getProperties());
      }



      //Parameters

      for (Parameter parameter : testsuite.getParameters())
      {
         javaBuilder.addParameter(parameter.getName(), Arrays.asList(parameter.getValues()));
      }

      javaBuilder.updateFQNParameters(System.getProperties());

   }
}
