/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/

package org.jboss.portal.test.framework.deployment;

import org.jboss.logging.Logger;
import org.jboss.portal.test.framework.agent.TestAgent;
import org.jboss.portal.test.framework.server.Node;

import java.io.IOException;
import java.net.URL;

/**
 * A deployment in the portlet container.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 7954 $
 */
class Deployment
{

   private static final Logger log = Logger.getLogger(Deployment.class);

   /** The nodes we must deploy to. */
   private final Node[] nodes;

   /** The local url of the deployment to make. */
   private URL url;

   /** . */
   private boolean deployed;

   public Deployment(Node[] nodes, URL url)
   {
      if (nodes == null)
      {
         throw new IllegalArgumentException("Does not accept null argument");
      }
      if (url == null)
      {
         throw new IllegalArgumentException("Does not accept null argument");
      }
      this.nodes = nodes;
      this.url = url;
      this.deployed = false;
   }

   /**
    * Deploy a portlet application. This method blocks until the deployment is finished
    *
    * @throws Exception any exception that would make the deployment process fail
    * @throws IllegalStateException if the session if closed
    * @throws IllegalArgumentException if the application name does not give a valid deployment
    * @throws java.io.IOException
    */
   void deploy() throws Exception, IllegalArgumentException, IllegalStateException, IOException
   {
      if (deployed)
      {
         throw new IllegalStateException("Cannot make 2 deployment at the same time");
      }

      //
      for (int i = 0;i < nodes.length;i++)
      {
         Node node = nodes[i];
         TestAgent agent = (TestAgent)node.lookupService(TestAgent.SERVICE_ID);
         agent.deploy(url);
      }

      //
      deployed = true;
   }

   /** Undeploy a portlet application. This method blocks until the undeployment is finished. */
   void undeploy()
   {
      URL url = this.url;
      this.url = null;

      //
      for (int i = 0;i < nodes.length;i++)
      {
         try
         {
            Node node = nodes[i];
            TestAgent agent = (TestAgent)node.lookupService(TestAgent.SERVICE_ID);
            agent.undeploy(url);
         }
         catch (Exception e)
         {
            Deployment.log.warn("Unexpected exception when undeploying " + e);
         }
      }

      //
      deployed = false;
   }
}
