/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/

package org.jboss.portal.test.framework.embedded;

import org.jboss.portal.common.xml.XMLTools;
import org.jboss.portal.common.io.IOTools;
import org.jboss.resource.adapter.jdbc.local.LocalTxDataSource;
import org.jboss.resource.connectionmanager.CachedConnectionManagerReference;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.naming.InitialContext;
import javax.transaction.TransactionManager;
import java.io.InputStream;
import java.net.URL;
import java.sql.Connection;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

/**
 * An helper to setup JDNI available objects
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 7954 $
 */
public class DataSourceSupport
{

   //
   private TransactionManager transactionManager;

   //
   private CachedConnectionManagerReference connectionManagerReference;

   //
   private Config config;

   private String jndiName = "java:/DefaultDS";

   //
   private LocalTxDataSource ds;

   public DataSourceSupport(Config cfg)
   {
      this.config = cfg;
   }

   public DataSourceSupport()
   {
      this(new Config("", "", "", "", ""));
   }

   public void create()
   {
   }

   public void start() throws Exception
   {

      //
      ds = new LocalTxDataSource();
      ds.setConnectionURL(config.getConnectionURL());
      ds.setDriverClass(config.getDriverClass());
      ds.setUserName(config.getUserName());
      ds.setPassword(config.getPassword());
      ds.setJndiName(jndiName);
      ds.setMaxSize(10);
      ds.setMinSize(0);
      ds.setBlockingTimeout(1000);
      ds.setIdleTimeout(10000);
      ds.setTransactionManager(transactionManager);
      ds.setCachedConnectionManager(connectionManagerReference);
      ds.setInitialContextProperties(new Hashtable());
      ds.start();
   }

   public void stop()
   {
      // Make hsqldb flush its data on the disk
      if ("org.hsqldb.jdbcDriver".equals(config.driverClass))
      {
         try
         {
            javax.sql.DataSource ds = (javax.sql.DataSource)new InitialContext().lookup(jndiName);
            Connection conn = null;
            Statement st = null;
            try
            {
               conn = ds.getConnection();
               st = conn.createStatement();
               st.execute("SHUTDOWN");
            }
            finally
            {
               IOTools.safeClose(st);
               IOTools.safeClose(conn);
            }
         }
         catch (Exception e)
         {
            System.out.println("was not able to shutdown hsqldb");
            e.printStackTrace();
         }
      }

      ds = null;
   }

   public void destroy()
   {
   }

   public Config getConfig()
   {
      return config;
   }

   public void setConfig(Config config)
   {
      this.config = config;
   }

   public TransactionManager getTransactionManager()
   {
      return transactionManager;
   }

   public void setTransactionManager(TransactionManager transactionManager)
   {
      this.transactionManager = transactionManager;
   }

   public CachedConnectionManagerReference getConnectionManagerReference()
   {
      return connectionManagerReference;
   }

   public void setConnectionManagerReference(CachedConnectionManagerReference connectionManagerReference)
   {
      this.connectionManagerReference = connectionManagerReference;
   }

   public static class Config
   {

      /** . */
      private String name;

      /** . */
      private String connectionURL;

      /** . */
      private String driverClass;

      /** . */
      private String userName;

      /** . */
      private String password;

      public Config(String name, String connectionURL, String driverClass, String userName, String password)
      {
         this.name = name;
         this.connectionURL = connectionURL;
         this.driverClass = driverClass;
         this.userName = userName;
         this.password = password;
      }

      public Config()
      {

      }

      public String getName()
      {
         return name;
      }

      public String getConnectionURL()
      {
         return connectionURL;
      }

      public String getDriverClass()
      {
         return driverClass;
      }

      public String getUserName()
      {
         return userName;
      }

      public String getPassword()
      {
         return password;
      }

      public String toString()
      {
         return "Datasource[" + name + "]";
      }


      public void setName(String name)
      {
         this.name = name;
      }

      public void setConnectionURL(String connectionURL)
      {
         this.connectionURL = connectionURL;
      }

      public void setDriverClass(String driverClass)
      {
         this.driverClass = driverClass;
      }

      public void setUserName(String userName)
      {
         this.userName = userName;
      }

      public void setPassword(String password)
      {
         this.password = password;
      }

//      public static MultiValuedTestParameterValue fromXML2(URL url) throws Exception
//      {
//         Config[] configs = fromXML(url);
//         List list = Arrays.asList(configs);
//         return new MultiValuedTestParameterValue(list);
//      }

      public static Config[] fromXML(URL url) throws Exception
      {
         ArrayList configs = new ArrayList();
         InputStream in = null;
         try
         {
            in = IOTools.safeBufferedWrapper(url.openStream());
            Document doc = XMLTools.getDocumentBuilderFactory().newDocumentBuilder().parse(in);
            for (Iterator i = XMLTools.getChildrenIterator(doc.getDocumentElement(), "datasource"); i.hasNext();)
            {
               Element childElt = (Element)i.next();

               // Parse the datasource name, taking in account the deprecated display-name element
               Element nameElt = XMLTools.getUniqueChild(childElt, "datasource-name", false);
               if (nameElt == null)
               {
                  System.out.println("XML element datasource-name is not present, trying deprecated display-name element instead, you should fix your datasources.xml");
                  nameElt = XMLTools.getUniqueChild(childElt, "display-name", true);
               }

               // Parse the rest of the configuration
               Element connectionURLElt = XMLTools.getUniqueChild(childElt, "connection-url", true);
               Element driverClassElt = XMLTools.getUniqueChild(childElt, "driver-class", true);
               Element userNameElt = XMLTools.getUniqueChild(childElt, "user-name", true);
               Element passwordElt = XMLTools.getUniqueChild(childElt, "password", true);
               String name = XMLTools.asString(nameElt);
               String connectionURL = XMLTools.asString(connectionURLElt);
               String driverClass = XMLTools.asString(driverClassElt);
               String userName = XMLTools.asString(userNameElt);
               String password = XMLTools.asString(passwordElt);
               Config dsCfg = new Config(
                  name,
                  connectionURL,
                  driverClass,
                  userName,
                  password);
               configs.add(dsCfg);
            }
            return (Config[])configs.toArray(new Config[configs.size()]);
         }
         finally
         {
            IOTools.safeClose(in);
         }
      } 

      public static Config obtainConfig(String datasources, String dataSourceName) throws Exception
      {
         URL url = Thread.currentThread().getContextClassLoader().getResource(datasources);

         Config[] configs = fromXML(url);

         for (Config config : configs)
         {
            if (config.getName().equals(dataSourceName))
            {
               return config;
            }
         }

         throw new IllegalStateException("Could not obtain Config for {datasourceName:datasources} - {" + dataSourceName + ":" + datasources + "}" );

      }
   }



}
