/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.webbeans.integration.deployer.env;

import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.beans.metadata.spi.builder.ParameterMetaDataBuilder;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractSimpleRealDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.webbeans.bootstrap.api.Bootstrap;
import org.jboss.webbeans.bootstrap.api.Environments;
import org.jboss.webbeans.context.api.BeanStore;
import org.jboss.webbeans.context.api.helpers.ConcurrentHashMapBeanStore;
import org.jboss.webbeans.integration.deployer.DeployersUtils;

/**
 * Deploy WebBeans boostrap service.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class WebBeansBootstrapDeployer extends AbstractSimpleRealDeployer<WebBeanDiscoveryEnvironment>
{
   public WebBeansBootstrapDeployer()
   {
      super(WebBeanDiscoveryEnvironment.class);
      setTopLevelOnly(true);
      setStage(DeploymentStages.PRE_REAL);
      addOutput(BeanMetaData.class);
   }

   @Override
   public void deploy(DeploymentUnit unit, WebBeanDiscoveryEnvironment env) throws DeploymentException
   {
      String unitName = unit.getName();
      String envName = unitName + "_JBossWebBeanDiscovery";
      BeanMetaDataBuilder envWrapper = BeanMetaDataBuilder.createBuilder(envName, "org.jboss.webbeans.integration.deployer.env.JBossWebBeanDiscovery");
      envWrapper.addConstructorParameter(WebBeanDiscoveryEnvironment.class.getName(), env);
      unit.addAttachment(envName + "_" + BeanMetaData.class.getSimpleName(), envWrapper.getBeanMetaData());

      String bootstrapName = DeployersUtils.getBootstrapBeanName(unit);
      BeanMetaDataBuilder bootstrap = BeanMetaDataBuilder.createBuilder(bootstrapName, "org.jboss.webbeans.bootstrap.api.helpers.BootstrapBean");
      bootstrap.addConstructorParameter(Bootstrap.class.getName(), createBootstrap(unit));
      bootstrap.addPropertyMetaData("webBeanDiscovery", bootstrap.createInject(envName));
      ValueMetaData ejbServicesValue = createEjbConnector("JBossEjbServices", "org.jboss.webbeans.integration.ejb.JBossEjbServices", unit);
      bootstrap.addPropertyMetaData("ejbServices", ejbServicesValue);
      bootstrap.addPropertyMetaData("jpaServices", createEjbConnector("JBossJpaServices", "org.jboss.webbeans.integration.persistence.JBossJpaServices", unit));
      bootstrap.addPropertyMetaData("resourceServices", bootstrap.createInject("JBossResourceServices"));
      bootstrap.addPropertyMetaData("webServices", bootstrap.createInject("JBossWebServices"));
      bootstrap.addPropertyMetaData("jmsServices", bootstrap.createInject("JBossJmsServices"));
      bootstrap.addPropertyMetaData("transactionServices", bootstrap.createInject("JBossTransactionServices"));
      bootstrap.addPropertyMetaData("applicationContext", createBeanStore());
      bootstrap.addPropertyMetaData("environment", Environments.EE);
      bootstrap.setCreate("initialize");
      bootstrap.setStart("boot");
      bootstrap.setDestroy("shutdown");
      bootstrap.addDependency("RealTransactionManager"); // so we know TM is present in JBossTransactionServices
      // call dynamic dependency creator for EJBs
      ParameterMetaDataBuilder install = bootstrap.addInstallWithParameters("createDepenencies", "DynamicDependencyCreator", null, ControllerState.CONFIGURED);
      install.addParameterMetaData(Object.class.getName(), bootstrapName);
      install.addParameterMetaData(Iterable.class.getName(), bootstrap.createInject(ejbServicesValue.getUnderlyingValue(), "ejbContainerNames"));
      install.addParameterMetaData(String.class.getName(), "Start");
      install.addParameterMetaData(String.class.getName(), "Start");
      
      unit.addAttachment(bootstrapName + "_" + BeanMetaData.class.getSimpleName(), bootstrap.getBeanMetaData());      
   }

   /**
    * Create new bean store.
    *
    * @return the bean store instance
    */
   protected BeanStore createBeanStore()
   {
      return new ConcurrentHashMapBeanStore();
   }

   /**
    * Create ejb connector.
    *
    * @param name the connector name
    * @param bean the bean to create
    * @param unit the deployment unit
    * @return new inject metadata
    */
   protected ValueMetaData createEjbConnector(String name, String bean, DeploymentUnit unit)
   {
      String beanName = unit.getName() + "_" + name;
      BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(beanName, bean);
      builder.setFactory(name);
      builder.setFactoryMethod("createBean");
      builder.addPropertyMetaData("deploymentUnit", unit);
      unit.addAttachment(beanName + "_" + BeanMetaData.class.getSimpleName(), builder.getBeanMetaData());
      return builder.createInject(beanName);
   }

   /**
    * Create bootstrap bean.
    *
    * @param unit the deployment unit
    * @return new injected bootstrap metadata
    */
   protected ValueMetaData createBootstrap(DeploymentUnit unit)
   {
      String beanName = unit.getName() + "_WebBeansBootstrap";
      BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(beanName, "org.jboss.webbeans.bootstrap.WebBeansBootstrap");
      unit.addAttachment(beanName + "_" + BeanMetaData.class.getSimpleName(), builder.getBeanMetaData());
      return builder.createInject(beanName);
   }
}
