package org.jboss.webbeans.tck.unit.implementation.enterprise;

import javax.inject.DefinitionException;

import org.jboss.webbeans.tck.AbstractTest;
import org.jboss.webbeans.tck.impl.SpecAssertion;
import org.testng.annotations.Test;

/**
 * Sections
 * 
 * 3.3. Enterprise Web Beans 3.3.1. Which EJBs are enterprise Web Beans? 3.3.2.
 * API types of an enterprise Web Bean 3.3.3. Declaring an enterprise Web Bean
 * using annotations 3.3.4. Declaring an enterprise Web Bean using XML
 * 
 * @author Nicklas Karlsson
 * 
 * Spec version: PRD2
 */
public class EnterpriseBeanDeclarationTest extends AbstractTest
{

   /**
    * An EJB stateless session bean must belong to the @Dependent pseudo-scope.
    * If an enterprise Web Bean specifies an illegal scope, a
    * DefinitionException is thrown by the Web Bean manager at initialization
    * time
    */
   @Test(groups = { "enterpriseBeans" })
   @SpecAssertion(section = "3.3")
   public void testStatelessWithDependentScopeOK()
   {
      deployBeans(Giraffe.class);
   }

   /**
    * An EJB stateless session bean must belong to the @Dependent pseudo-scope.
    * If an enterprise Web Bean specifies an illegal scope, a
    * DefinitionException is thrown by the Web Bean manager at initialization
    * time
    */
   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testStatelessWithRequestScopeFails()
   {
      deployBeans(Beagle_Broken.class);
   }

   /**
    * An EJB stateless session bean must belong to the @Dependent pseudo-scope.
    * If an enterprise Web Bean specifies an illegal scope, a
    * DefinitionException is thrown by the Web Bean manager at initialization
    * time
    */

   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testStatelessWithConversationScopeFails()
   {
      deployBeans(Boxer_Broken.class);
   }

   /**
    * An EJB stateless session bean must belong to the @Dependent pseudo-scope.
    * If an enterprise Web Bean specifies an illegal scope, a
    * DefinitionException is thrown by the Web Bean manager at initialization
    * time
    */
   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testStatelessWithSessionScopeFails()
   {
      deployBeans(Bullmastiff_Broken.class);
   }

   /**
    * An EJB stateless session bean must belong to the @Dependent pseudo-scope.
    * If an enterprise Web Bean specifies an illegal scope, a
    * DefinitionException is thrown by the Web Bean manager at initialization
    * time
    */
   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testStatelessWithApplicationScopeFails()
   {
      deployBeans(Dachshund_Broken.class);
   }

   /**
    * An EJB singleton bean must belong to either the @ApplicationScoped scope
    * or to the @Dependent pseudo-scope. If an enterprise Web Bean specifies an
    * illegal scope, a DefinitionException is thrown by the Web Bean manager at
    * initialization time
    */
   @Test(groups = { "enterpriseBeans" })
   @SpecAssertion(section = "3.3")
   public void testSingletonWithDependentScopeOK()
   {
      deployBeans(Labrador.class);
   }

   /**
    * An EJB singleton bean must belong to either the @ApplicationScoped scope
    * or to the @Dependent pseudo-scope. If an enterprise Web Bean specifies an
    * illegal scope, a DefinitionException is thrown by the Web Bean manager at
    * initialization time
    */
   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testSingletonWithRequestScopeFails()
   {
      deployBeans(Greyhound_Broken.class);
      assert false;
   }

   /**
    * An EJB singleton bean must belong to either the @ApplicationScoped scope
    * or to the @Dependent pseudo-scope. If an enterprise Web Bean specifies an
    * illegal scope, a DefinitionException is thrown by the Web Bean manager at
    * initialization time
    */
   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testSingletonWithConversationScopeFails()
   {
      deployBeans(Husky_Broken.class);
      assert false;
   }

   /**
    * An EJB singleton bean must belong to either the @ApplicationScoped scope
    * or to the @Dependent pseudo-scope. If an enterprise Web Bean specifies an
    * illegal scope, a DefinitionException is thrown by the Web Bean manager at
    * initialization time
    */
   @Test(groups = { "enterpriseBeans" }, expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testSingletonWithSessionScopeFails()
   {
      deployBeans(IrishTerrier_Broken.class);
   }

   /**
    * An EJB singleton bean must belong to either the @ApplicationScoped scope
    * or to the @Dependent pseudo-scope. If an enterprise Web Bean specifies an
    * illegal scope, a DefinitionException is thrown by the Web Bean manager at
    * initialization time
    */
   @Test(groups = { "enterpriseBeans" })
   @SpecAssertion(section = "3.3")
   public void testSingletonWithApplicationScopeOK()
   {
      deployBeans(Laika.class);
   }

   /**
    * If the implementation class of an enterprise Web Bean is annotated @Interceptor
    * or @Decorator, a DefinitionException is thrown by the Web Bean manager at
    * initialization time.
    */
   @Test(expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testEnterpriseBeanInterceptorFails()
   {
      deployBeans(Pug_Broken.class);
   }

   /**
    * If the implementation class of an enterprise Web Bean is annotated @Interceptor
    * or @Decorator, a DefinitionException is thrown by the Web Bean manager at
    * initialization time.
    */
   @Test(expectedExceptions = DefinitionException.class)
   @SpecAssertion(section = "3.3")
   public void testEnterpriseBeanDecoratorFails()
   {
      deployBeans(Pekingese_Broken.class);
   }

   /**
    * Only one Web Bean per implementation class may be defined using
    * annotations.
    */
   @Test(expectedExceptions = DefinitionException.class, groups = { "enterpriseBeans", "stub" })
   @SpecAssertion(section = "3.3")
   public void testMultipleAnnotationDefinedEnterpriseBeansWithSameImplementationClassFails()
   {
      // TODO: testable?
      assert false;
   }

   /**
    * Note that multiple enterprise Web Beans may share the same implementation
    * class. This occurs when Web Beans are defined using XML
    */
   @Test(groups = { "webbeansxml", "enterpriseBeans", "stub" })
   @SpecAssertion(section = "3.3")
   public void testMultipleXMLDefinedEnterpriseBeansWithSameImplementationClassOK()
   {
      assert false;
   }

   /**
    * All session beans exposing an EJB 3.x client view and declared via an EJB
    * component defining annotation on the EJB bean class are Web Beans, and
    * thus no special declaration is required.
    */
   @Test(groups = { "enterpriseBeans", "stub" })
   @SpecAssertion(section = "3.3.2")
   public void testAnnotatedEnterpriseBean()
   {
      // TODO: dupe?
      assert false;
   }

   /**
    * Additional enterprise Web Beans for these EJBs may be defined using XML,
    * by specifying the bean class in web-beans.xml.
    */
   @Test(groups = { "enterpriseBeans", "webbeansxml", "stub" })
   @SpecAssertion(section = "3.3.2")
   public void testAnnotatedEnterpriseBeanComplementedWithXML()
   {
      // TODO dupe?
      assert false;
   }

   /**
    * All session beans exposing an EJB 3.x client view and declared in
    * ejb-jar.xml are also Web Beans.
    */
   @Test(groups = { "enterpriseBeans", "ejbjarxml", "stub" })
   @SpecAssertion(section = "3.3.2")
   public void testEJBJARDefinedEnterpriseBean()
   {
      // TODO dupe?
      assert false;
   }

   /**
    * Additional enterprise Web Beans for these EJBs may be defined using XML,
    * by specifying the bean class and EJB name in web-beans.xml
    */
   @Test(groups = { "enterpriseBeans", "ejbjarxml", "webbeansxml", "stub" })
   @SpecAssertion(section = "3.3.2")
   public void testEJBJARDefinedEnterpriseBeanComplementedWithXML()
   {
      // TODO dupe?
      assert false;
   }

   /**
    * The set of API types for an enterprise Web Bean contains all local
    * interfaces of the bean that do not have wildcard type parameters or type
    * variables and their superinterfaces
    */
   @Test(groups = { "enterpriseBeans", "stub" })
   @SpecAssertion(section = "3.3.3")
   public void testAPITypesAreLocalInterfacesWithoutWildcardTypesOrTypeVariablesWithSuperInterfaces()
   {
      assert false;
   }

   /**
    * If the EJB bean has a bean class local view and the bean class is not a
    * parameterized type, the set of API types contains the bean class and all
    * superclasses
    */
   @Test(groups = { "enterpriseBeans", "stub" })
   @SpecAssertion(section = "3.3.3")
   public void testEnterpriseBeanWithLocalViewAndParameterizedTypeIncludesBeanClassAndSuperclassesInAPITypes()
   {
      assert false;
   }

   /**
    * In addition, java.lang.Object is an API type of every enterprise Web Bean.
    */
   @Test(groups = "enterpriseBeans")
   @SpecAssertion(section = "3.3.3")
   public void testObjectIsInAPITypes()
   {
      assert createEnterpriseBean(Laika.class).getTypes().contains(Object.class);
   }

   /**
    * Remote interfaces are not included in the set of API types.
    */
   @Test(groups = { "enterpriseBeans", "stub" })
   @SpecAssertion(section = "3.3.3")
   public void testRemoteInterfacesAreNotInAPITypes()
   {
      assert false;
   }

   /**
    * Enterprise Web Beans may be declared in web-beans.xml using the bean class
    * name (for EJBs defined using a component- defining annotation) or bean
    * class and EJB name (for EJBs defined in ejb-jar.xml). The ejbName
    * attribute declares the EJB name of an EJB defined in ejb-jar.xml
    */
   @Test(groups = { "enterpriseBeans", "webbeansxml", "ejbjarxml", "stub" })
   @SpecAssertion(section = "3.3")
   public void testXMLFilesEJBNameUsage()
   {
      assert false;
   }

   /**
    * Enterprise Web Beans may not be message-driven beans. If an enterprise Web
    * Bean declared in XML is a message-driven bean, a DefinitionException is
    * thrown by the Web Bean manager at initialization time.
    */
   @Test(expectedExceptions = DefinitionException.class, groups = { "enterpriseBeans" })
   @SpecAssertion(section = "3.3")
   public void testMessageDrivenBeansNotOK()
   {
      deployBeans(Leopard.class);
   }

   /**
    * The default name for an enterprise Web Bean is the unqualified class name
    * of the Web Bean implementation class, after converting the first character
    * to lower case.
    */
   @Test(groups = "enterpriseBeans")
   @SpecAssertion(section = "3.3.8")
   public void testDefaultName()
   {
      assert createEnterpriseBean(Pitbull.class).getName().equals("pitbull");
   }

}
