/*
 * JBoss, Home of Professional Open Source
 * Copyright 2014, Red Hat, Inc., and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.weld.config;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.security.AccessController;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import org.jboss.weld.bootstrap.api.Service;
import org.jboss.weld.bootstrap.spi.BeanDeploymentArchive;
import org.jboss.weld.bootstrap.spi.BootstrapConfiguration;
import org.jboss.weld.bootstrap.spi.Deployment;
import org.jboss.weld.configuration.spi.ExternalConfiguration;
import org.jboss.weld.exceptions.IllegalStateException;
import org.jboss.weld.logging.BeanLogger;
import org.jboss.weld.logging.ConfigurationLogger;
import org.jboss.weld.resources.spi.ResourceLoader;
import org.jboss.weld.resources.spi.ResourceLoadingException;
import org.jboss.weld.security.GetSystemPropertyAction;
import org.jboss.weld.util.Preconditions;
import org.jboss.weld.util.collections.ImmutableMap;

/**
 * Represents an immutable per-deployment Weld configuration.
 *
 * <p>
 * Each property may be set in three different sources:
 * </p>
 * <ol>
 * <li>Bootstrap configuration provided by an integrator</li>
 * <li>System property</li>
 * <li>Properties file <code>weld.properties</code></li>
 * </ol>
 *
 * <p>
 * For backwards compatibility:
 * </p>
 * <ul>
 * <li>properties files <code>org.jboss.weld.executor.properties</code> and <code>org.jboss.weld.bootstrap.properties</code> are also loaded for some
 * configuration keys,</li>
 * <li>some system properties with obsolete keys are considered</li>
 * </ul>
 *
 * @author Martin Kouba
 * @see BootstrapConfiguration
 * @see ConfigurationKey
 */
@SuppressWarnings("deprecation")
public class WeldConfiguration implements Service {

    public static final String CONFIGURATION_FILE = "weld.properties";

    private static final String EXECUTOR_CONFIGURATION_FILE = "org.jboss.weld.executor.properties";

    private static final String BOOTSTRAP_CONFIGURATION_FILE = "org.jboss.weld.bootstrap.properties";

    private final Map<ConfigurationKey, Object> properties;

    private final File proxyDumpFilePath;

    /**
     *
     * @param bootstrapConfiguration
     * @param deployment
     */
    public WeldConfiguration(BootstrapConfiguration bootstrapConfiguration, ExternalConfiguration externalConfiguration, Deployment deployment) {
        Preconditions.checkArgumentNotNull(deployment, "deployment");
        this.properties = init(bootstrapConfiguration, externalConfiguration, deployment);
        this.proxyDumpFilePath = initProxyDumpFilePath();
        ConfigurationLogger.LOG.configurationInitialized(properties);
    }

    /**
     *
     * @param key
     * @return the property for the given key
     * @throws IllegalStateException If the property type does not match the required type
     */
    public String getStringProperty(ConfigurationKey key) {
        return getProperty(key, String.class);
    }

    /**
     *
     * @param key
     * @return the property for the given key
     * @throws IllegalStateException If the property type does not match the required type
     */
    public Boolean getBooleanProperty(ConfigurationKey key) {
        return getProperty(key, Boolean.class);
    }

    /**
     *
     * @param key
     * @return the property for the given key
     * @throws IllegalStateException If the property type does not match the required type
     */
    public Long getLongProperty(ConfigurationKey key) {
        return getProperty(key, Long.class);
    }

    /**
     *
     * @param key
     * @return the property for the given key
     * @throws IllegalStateException If the property type does not match the required type
     */
    public Integer getIntegerProperty(ConfigurationKey key) {
        return getProperty(key, Integer.class);
    }

    /**
     *
     * @return the path or <code>null</code> if the generated bytecode should not be dumped
     * @see ConfigurationKey#PROXY_DUMP
     */
    public File getProxyDumpFilePath() {
        return proxyDumpFilePath;
    }

    @Override
    public void cleanup() {
        if (properties != null) {
            properties.clear();
        }
    }

    static void merge(Map<ConfigurationKey, Object> original, Map<ConfigurationKey, Object> toMerge) {
        for (Entry<ConfigurationKey, Object> entry : toMerge.entrySet()) {
            Object existing = original.get(entry.getKey());
            if (existing != null) {
                ConfigurationLogger.LOG.configurationKeyAlreadySet(entry.getKey(), existing, entry.getValue());
            } else {
                original.put(entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     *
     * @param key
     * @param requiredType
     * @throws IllegalStateException If the configuration property type does not match the required type
     */
    static void checkRequiredType(ConfigurationKey key, Class<?> requiredType) {
        if (!key.isValidValueType(requiredType)) {
            throw ConfigurationLogger.LOG.configurationPropertyTypeMismatch(key.getDefaultValue().getClass(), requiredType);
        }
    }

    /**
     *
     * @param key
     * @return the string value of the system property or <code>null</code>
     */
    static String getSystemProperty(String key) {
        try {
            return AccessController.doPrivileged(new GetSystemPropertyAction(key));
        } catch (Throwable ignore) {
            return null;
        }
    }

    private Map<ConfigurationKey, Object> init(BootstrapConfiguration bootstrapConfiguration, ExternalConfiguration externalConfiguration, Deployment deployment) {
        Map<ConfigurationKey, Object> properties = new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);

        // 1. Properties files
        // weld.properties
        merge(properties, readFileProperties(findPropertiesFiles(deployment, CONFIGURATION_FILE)));
        // org.jboss.weld.bootstrap.properties
        merge(properties,
                readObsoleteFileProperties(
                        findPropertiesFiles(deployment, BOOTSTRAP_CONFIGURATION_FILE),
                        ImmutableMap.<String, ConfigurationKey>builder().put("concurrentDeployment", ConfigurationKey.CONCURRENT_DEPLOYMENT)
                                .put("preloaderThreadPoolSize", ConfigurationKey.PRELOADER_THREAD_POOL_SIZE).build()));
        // org.jboss.weld.executor.properties
        merge(properties,
                readObsoleteFileProperties(
                        findPropertiesFiles(deployment, EXECUTOR_CONFIGURATION_FILE),
                        ImmutableMap.<String, ConfigurationKey>builder().put("threadPoolSize", ConfigurationKey.EXECUTOR_THREAD_POOL_SIZE)
                                .put("threadPoolDebug", ConfigurationKey.EXECUTOR_THREAD_POOL_DEBUG)
                                .put("threadPoolType", ConfigurationKey.EXECUTOR_THREAD_POOL_TYPE)
                                .put("threadPoolKeepAliveTime", ConfigurationKey.EXECUTOR_THREAD_POOL_KEEP_ALIVE_TIME).build()));

        // 2. System properties
        merge(properties, getSystemProperties());
        merge(properties, getObsoleteSystemProperties());

        // 3. Integrator SPI
        // ExternalConfiguration.getConfigurationProperties() map has precedence
        merge(properties, processExternalConfiguration(externalConfiguration));
        merge(properties, processBootstrapConfiguration(bootstrapConfiguration));

        return properties;
    }

    private File initProxyDumpFilePath() {
        String dumpPath = getStringProperty(ConfigurationKey.PROXY_DUMP);
        if (!dumpPath.isEmpty()) {
            File tmp = new File(dumpPath);
            if (!tmp.isDirectory() && !tmp.mkdirs()) {
                BeanLogger.LOG.directoryCannotBeCreated(tmp.toString());
                return null;
            } else {
                return tmp;
            }
        }
        return null;
    }

    @edu.umd.cs.findbugs.annotations.SuppressWarnings(value = "DMI_COLLECTION_OF_URLS", justification = "Only local URLs involved")
    private Set<URL> findPropertiesFiles(Deployment deployment, String fileName) {
        Set<ResourceLoader> resourceLoaders = new HashSet<ResourceLoader>();
        Set<URL> files = new HashSet<URL>();

        ResourceLoader deploymentResourceLoader = deployment.getServices().get(ResourceLoader.class);
        if (deploymentResourceLoader != null) {
            resourceLoaders.add(deploymentResourceLoader);
        }

        for (BeanDeploymentArchive archive : deployment.getBeanDeploymentArchives()) {
            ResourceLoader resourceLoader = archive.getServices().get(ResourceLoader.class);
            if (resourceLoader == null) {
                ConfigurationLogger.LOG.resourceLoaderNotSpecifiedForArchive(archive);
                continue;
            }
            resourceLoaders.add(resourceLoader);
        }
        for (ResourceLoader resourceLoader : resourceLoaders) {
            URL file = resourceLoader.getResource(fileName);
            if (file != null) {
                files.add(file);
            }
        }
        return files;
    }

    /**
     *
     * @return all the properties set as system properties
     */
    private Map<ConfigurationKey, Object> getSystemProperties() {
        Map<ConfigurationKey, Object> found = new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);
        for (ConfigurationKey key : ConfigurationKey.values()) {
            String property = getSystemProperty(key.get());
            if (property != null) {
                found.put(key, key.convertValue(property));
            }
        }
        return found;
    }

    /**
     *
     * @return all the properties whose system property keys were different in previous versions
     */
    private Map<ConfigurationKey, Object> getObsoleteSystemProperties() {
        Map<ConfigurationKey, Object> found = new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);
        String concurrentDeployment = getSystemProperty("org.jboss.weld.bootstrap.properties.concurrentDeployment");
        if (concurrentDeployment != null) {
            found.put(ConfigurationKey.CONCURRENT_DEPLOYMENT, ConfigurationKey.CONCURRENT_DEPLOYMENT.convertValue(concurrentDeployment));
        }
        String preloaderThreadPoolSize = getSystemProperty("org.jboss.weld.bootstrap.properties.preloaderThreadPoolSize");
        if (preloaderThreadPoolSize != null) {
            found.put(ConfigurationKey.PRELOADER_THREAD_POOL_SIZE, ConfigurationKey.PRELOADER_THREAD_POOL_SIZE.convertValue(preloaderThreadPoolSize));
        }
        return found;
    }

    /**
     *
     * @param resourceLoader
     * @return all the properties from the weld.properties file
     */
    @edu.umd.cs.findbugs.annotations.SuppressWarnings(value = "DMI_COLLECTION_OF_URLS", justification = "Only local URLs involved")
    private Map<ConfigurationKey, Object> readFileProperties(Set<URL> files) {
        if (files.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<ConfigurationKey, Object> found = new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);
        for (URL file : files) {
            ConfigurationLogger.LOG.readingPropertiesFile(file);
            Properties fileProperties = loadProperties(file);
            for (String name : fileProperties.stringPropertyNames()) {
                handleStringKey(found, name, fileProperties.getProperty(name));
            }
        }
        return found;
    }

    @edu.umd.cs.findbugs.annotations.SuppressWarnings(value = "DMI_COLLECTION_OF_URLS", justification = "Only local URLs involved")
    private Map<ConfigurationKey, Object> readObsoleteFileProperties(Set<URL> files, Map<String, ConfigurationKey> nameToKeyMap) {
        if (files.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<ConfigurationKey, Object> found = new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);
        for (URL file : files) {
            Properties fileProperties = loadProperties(file);
            for (String name : fileProperties.stringPropertyNames()) {
                ConfigurationKey key = nameToKeyMap.get(name);
                if (key != null) {
                    Object value = key.convertValue(fileProperties.getProperty(name));
                    Object previous = found.put(key, value);
                    if (previous != null && !previous.equals(value)) {
                        throw ConfigurationLogger.LOG.configurationKeyHasDifferentValues(key, previous, value);
                    }
                } else {
                    ConfigurationLogger.LOG.unsupportedConfigurationKeyFound(name + " in " + fileProperties);
                }
            }
        }
        return found;
    }

    private Map<ConfigurationKey, Object> processBootstrapConfiguration(BootstrapConfiguration bootstrapConfiguration) {
        if (bootstrapConfiguration != null) {

            Map<ConfigurationKey, Object> found =  new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);

            handleKey(found, ConfigurationKey.CONCURRENT_DEPLOYMENT, bootstrapConfiguration.isConcurrentDeploymentEnabled());
            handleKey(found, ConfigurationKey.PRELOADER_THREAD_POOL_SIZE, bootstrapConfiguration.getPreloaderThreadPoolSize());
            handleKey(found, ConfigurationKey.NON_PORTABLE_MODE, bootstrapConfiguration.isNonPortableModeEnabled());
        }
        return Collections.emptyMap();
    }

    private Map<ConfigurationKey, Object> processExternalConfiguration(ExternalConfiguration externalConfiguration) {
        if (externalConfiguration == null) {
            return Collections.emptyMap();
        }
        Map<ConfigurationKey, Object> found =  new EnumMap<ConfigurationKey, Object>(ConfigurationKey.class);
        for (Entry<String, Object> entry : externalConfiguration.getConfigurationProperties().entrySet()) {
            handleStringKey(found, entry.getKey(), entry.getValue());
        }
        return found;
    }

    /**
     *
     * @param properties
     * @param key
     * @param value
     */
    private void handleKey(Map<ConfigurationKey, Object> properties, ConfigurationKey key, Object value) {
        if (key.isValidValue(value)) {
            Object previous = properties.put(key, value);
            if (previous != null && !previous.equals(value)) {
                throw ConfigurationLogger.LOG.configurationKeyHasDifferentValues(key, previous, value);
            }
        } else {
            throw ConfigurationLogger.LOG.invalidConfigurationPropertyValue(value, key);
        }
    }

    /**
     *
     * @param properties
     * @param stringKey
     * @param value
     */
    private void handleStringKey(Map<ConfigurationKey, Object> properties, String stringKey, Object value) {
        ConfigurationKey key = ConfigurationKey.fromString(stringKey);
        if (key != null) {
            handleKey(properties, key, (value instanceof String ? key.convertValue((String) value) : value));
        } else {
            ConfigurationLogger.LOG.unsupportedConfigurationKeyFound(stringKey);
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T getProperty(ConfigurationKey key, Class<T> requiredType) {
        checkRequiredType(key, requiredType);
        Object property = properties.get(key);
        return (T) (property != null ? property : key.getDefaultValue());
    }

    private Properties loadProperties(URL url) {
        Properties properties = new Properties();
        try {
            properties.load(url.openStream());
        } catch (IOException e) {
            throw new ResourceLoadingException(e);
        }
        return properties;
    }

}
