/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.weld.manager.api;

import java.io.Serializable;

import javax.enterprise.context.spi.Contextual;
import javax.enterprise.inject.Instance;
import javax.enterprise.inject.spi.AnnotatedType;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.InjectionTarget;

import org.jboss.weld.bootstrap.api.ServiceRegistry;
import org.jboss.weld.construction.api.WeldCreationalContext;
import org.jboss.weld.ejb.spi.EjbDescriptor;
import org.jboss.weld.serialization.spi.BeanIdentifier;

/**
 * Functionality provided by the Weld Manager over and above the {@link BeanManager}.
 *
 * @author Pete Muir
 */
public interface WeldManager extends BeanManager, Serializable {

    /**
     * The injection target for the given EJB, or null if Weld was not given this descriptor in the deployment.
     *
     * This should only be used to create an inject contextual EJBs by the EJB container.
     * {@link #fireProcessInjectionTarget(AnnotatedType)} and
     * {@link #fireProcessInjectionTarget(AnnotatedType, InjectionTarget)}
     * must be used to obtain an {@link InjectionTarget} for non-contextual EJB injection.
     *
     * @param <T> the type of the EJB
     * @param descriptor the given EJB descriptor
     * @return the injection target for the given EJB
     */
    <T> InjectionTarget<T> createInjectionTarget(EjbDescriptor<T> descriptor);

    /**
     * Get the Bean object for the given EJB, or null if Weld was not given this descriptor in the deployment.
     *
     * @param <T> the type of the bean
     * @param descriptor the given EJB descriptor
     * @return the Bean object for the given EJB
     */
    <T> Bean<T> getBean(EjbDescriptor<T> descriptor);

    /**
     * Get the EjbDescriptor for the given EJB name
     *
     * @param <T> the type of the EJB
     * @param ejbName the given EJB name
     * @return the EjbDescriptor for the given EJB name
     */
    <T> EjbDescriptor<T> getEjbDescriptor(String ejbName);

    /**
     * Get the services registered for this manager
     *
     * @return the services registered for this manager
     */
    ServiceRegistry getServices();

    /**
     * Fire a ProcessInjectionTarget event for the given type.
     *
     * A helper method to allow integration code to easily fire the ProcessInjectionTarget for Java EE component classes
     * supporting injection
     *
     * The container must use the returned InjectionTarget to inject the Java EE component.
     *
     * @param <X> the type
     * @param type the type
     * @return final version of InjectionTarget after all observers have been invoked
     */
    <X> InjectionTarget<X> fireProcessInjectionTarget(AnnotatedType<X> type);

    /**
     * Fire a ProcessInjectionTarget event for the given type.
     *
     * A helper method to allow integration code to easily fire the ProcessInjectionTarget for Java EE component classes
     * supporting injection
     *
     * The container must use the returned InjectionTarget to inject the Java EE component.
     *
     * @param <X> the type
     * @param annotatedType the type
     * @param injectionTarget the injection target
     * @return final version of InjectionTarget after all observers have been invoked
     */
    <X> InjectionTarget<X> fireProcessInjectionTarget(AnnotatedType<X> annotatedType, InjectionTarget<X> injectionTarget);

    /**
     * The ID of the manager, identical to the ID provided by the BDA
     *
     * @return the ID of the manager
     */
    String getId();

    /**
     * TODO There is an open clarification issue CDI-671 which might affect the API design. I.e. this method might be changed in a way that breaks backward
     * compatibility in a final version of API.
     *
     * @seeIssue CDI-671
     * @return a new {@link Instance} with required type {@link Object} and no required qualifiers ({@link javax.enterprise.inject.Default} is added automatically during resolution if
     *         no qualifiers are selected)
     */
    Instance<Object> instance();

    @Override
    <T> WeldInjectionTargetFactory<T> getInjectionTargetFactory(AnnotatedType<T> type);

    @Override
    <T> WeldCreationalContext<T> createCreationalContext(Contextual<T> contextual);

    Bean<?> getPassivationCapableBean(BeanIdentifier identifier);

    /**
     * Returns a new instance of {@link WeldInjectionTargetBuilder} which can be used to create a new {@link WeldInjectionTarget} for the specified type.
     * @param <T> the type
     * @param type the specified type
     * @return a new {@link WeldInjectionTargetBuilder} instance for the specified type
     */
    <T> WeldInjectionTargetBuilder<T> createInjectionTargetBuilder(AnnotatedType<T> type);

    /**
     *
     * @return an unwrapped instance (e.g. the delegate in the case of forwarding implementation)
     */
    WeldManager unwrap();

    /**
     * Obtain an {@link AnnotatedType} that may be used to read the annotations of the given class or interface.
     * <p>
     * Allows multiple annotated types, based on the same underlying type, to be created. {@link AnnotatedType}s discovered by the container use the fully
     * qualified class name to identify the type.
     * <p>
     * This method must only be used when creating non-contextual instances, such as Java EE components. It's not designed to work with contextual instances.
     * <p>
     * If called after the container bootstrap finished, the client code is required to explicitly call {@link #disposeAnnotatedType(Class, String)}
     * as soon as the specified type should be garbage-collected (to avoid memory leaks).
     *
     * @param type
     * @param id
     * @return the {@link AnnotatedType}
     */
    <T> AnnotatedType<T> createAnnotatedType(Class<T> type, String id);

    /**
     * Dispose the {@link AnnotatedType} created for the identified type.
     * <p>
     * This method should be explicitly called for each result of {@link #createAnnotatedType(Class, String)} created after the container bootstrap finished.
     * <p>
     * It's not necessary to call this method unless the identified type should be a subject of garbage collection.
     *
     * @param type
     * @param id
     * @see #createAnnotatedType(Class, String)
     */
    <T> void disposeAnnotatedType(Class<T> type, String id);

}
