/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ws.common.injection.finders;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.ResourceBundle;

import org.jboss.ws.api.util.BundleUtils;
import org.jboss.ws.common.injection.InjectionException;
import org.jboss.ws.common.reflection.FieldFinder;
import org.jboss.wsf.spi.metadata.injection.InjectionMetaData;

/**
 * Lookups field that matches descriptor specified injection metadata.
 *
 * @author <a href="mailto:richard.opalka@jboss.org">Richard Opalka</a>
 */
public final class InjectionFieldFinder
extends FieldFinder
{
   private static final ResourceBundle bundle = BundleUtils.getBundle(InjectionFieldFinder.class);

   /**
    * Descriptor injection metadata.
    */
   private final InjectionMetaData injectionMD;

   /**
    * Constructor.
    *
    * @param injectionMD descriptor injection metadata
    */
   public InjectionFieldFinder(final InjectionMetaData injectionMD)
   {
      if (injectionMD == null)
         throw new IllegalArgumentException(BundleUtils.getMessage(bundle, "INJECTION_METADATA_CANNOT_BE_NULL"));

      this.injectionMD = injectionMD;
   }

   @Override
   public boolean matches(final Field field)
   {
      if (field.getName().equals(injectionMD.getTargetName()))
      {
         if (injectionMD.getValueClass() != null)
         {
            final Class<?> expectedClass = injectionMD.getValueClass();
            final Class<?> fieldClass = field.getType();

            return expectedClass.equals(fieldClass);
         }
         else
         {
            return true;
         }
      }

      return false;
   }

   @Override
   public void validate(final Collection<Field> fields)
   {
      super.validate(fields);

      if (fields.size() > 2)
      {
         throw new InjectionException(BundleUtils.getMessage(bundle, "MORE_THAN_ONE_FIELD_FOUND_MATCHING_THE_CRITERIA",  injectionMD));
      }
   }

   @Override
   public void validate(final Field field)
   {
      super.validate(field);

      ReflectionUtils.assertNotVoidType(field);
      ReflectionUtils.assertNotStatic(field);
      ReflectionUtils.assertNotFinal(field);
      ReflectionUtils.assertNotPrimitiveType(field);
   }

}
