/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.stack.metro;

import org.jboss.wsf.common.integration.WSConstants;
import org.jboss.wsf.spi.deployment.Deployment;
import org.jboss.wsf.spi.deployment.DeploymentAspect;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.stack.metro.metadata.sunjaxws.DDEndpoint;
import org.jboss.wsf.stack.metro.metadata.sunjaxws.DDEndpoints;
import org.jboss.wsf.stack.metro.metadata.RuntimeModelDeploymentAspect;

import javax.xml.ws.BindingType;
import javax.xml.ws.soap.SOAPBinding;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * An aspect that generates sun-jaxws.xml 
 *
 * @author Thomas.Diesler@jboss.org
 * @since 10-May-2007
 */
public class SunJaxwsDeploymentAspect extends DeploymentAspect
{
   @Override
   @SuppressWarnings("unchecked")
   public void start(Deployment dep)
   {
      DDEndpoints ddEndpoints = dep.getAttachment(DDEndpoints.class);
      if (ddEndpoints == null)
      {
         ddEndpoints = new DDEndpoints();
         for (Endpoint endpoint : dep.getService().getEndpoints())
         {
            String endpointName = endpoint.getShortName();
            String targetBean = endpoint.getTargetBeanName();
            String urlPattern = endpoint.getURLPattern();

            DDEndpoint ddEndpoint = new DDEndpoint(endpointName, targetBean, urlPattern);
            Class beanClass = endpoint.getTargetBeanClass();
            BindingType bindingType = (BindingType)beanClass.getAnnotation(BindingType.class);
            
            if (bindingType != null && bindingType.value().length() > 0)
            {
               String binding = bindingType.value();
               ddEndpoint.setBinding(binding);
               ddEndpoint.setEnableMTOM(isMtomEnabled(binding));
            }

            log.info("Add " + ddEndpoint);
            ddEndpoints.addEndpoint(ddEndpoint);
         }

         dep.addAttachment(DDEndpoints.class, ddEndpoints);
         putSunJaxwsConfigToDeployment(dep, ddEndpoints.createFileURL());
      }
   }

   @Override
   public void stop(Deployment dep)
   {
      DDEndpoints dd = dep.getAttachment(DDEndpoints.class);
      if (dd != null)
      {
         dd.destroyFileURL();
      }
   }
   
   /**
    * Puts Metro config file reference to the stack specific context properties. 
    *
    * @param dep webservice deployment
    * @param jaxwsConfigURL Metro DD URL
    * @see org.jboss.wsf.common.integration.WSConstants.STACK_CONTEXT_PARAMS
    */
   private static void putSunJaxwsConfigToDeployment(Deployment dep, URL jaxwsConfigURL)
   {
      // get property map
      Map<String, String> contextParams = (Map<String, String>)dep.getProperty(WSConstants.STACK_CONTEXT_PARAMS);
      if (contextParams == null)
      {
         // if there's no associated map with the property create it now
         contextParams = new HashMap<String, String>();
         dep.setProperty(WSConstants.STACK_CONTEXT_PARAMS, contextParams);
      }
      // put sun jaxws config URL to the property map
      contextParams.put(RuntimeModelDeploymentAspect.PARAM_SUN_JAXWS_URL, jaxwsConfigURL.toExternalForm());
   }
   
   private static boolean isMtomEnabled(String binding)
   {
      return binding.equals(SOAPBinding.SOAP11HTTP_MTOM_BINDING) || binding.equals(SOAPBinding.SOAP12HTTP_MTOM_BINDING);
   }
}
