/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.stack.metro.log;

import com.sun.xml.ws.api.message.Packet;
import com.sun.xml.ws.api.pipe.Pipe;
import com.sun.xml.ws.api.pipe.PipeCloner;
import com.sun.xml.ws.api.pipe.helper.AbstractFilterPipeImpl;
import org.jboss.logging.Logger;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.lang.reflect.Constructor;

/**
 * {@link Pipe} that dumps messages that pass through.
 * Borrowed from the SUN-RI sources.
 *
 * @author Kohsuke Kawaguchi
 * @author Heiko Braun
 */
public class DumpPipe extends AbstractFilterPipeImpl
{
   private static Logger msgLog = Logger.getLogger("org.jboss.wsf.spi.MessageTrace");

   private final String name;
   private final PrintStream out;
   private final XMLOutputFactory staxOut;

   /**
    * @param name
    *      Specify the name that identifies this {@link DumpPipe}
    *      instance. This string will be printed when this pipe
    *      dumps messages, and allows people to distinguish which
    *      pipe instance is dumping a message when multiple
    *      {@link DumpPipe}s print messages out.
    * @param out the output to send dumps to.
    * @param next the next {@link Pipe} in the pipeline.
    */
   public DumpPipe(String name, PrintStream out, Pipe next)
   {
      super(next);
      this.name = name;
      this.out = out;
      this.staxOut = XMLOutputFactory.newInstance();
   }

   /**
    * @param out the output to send dumps to.
    * @param next the next {@link Pipe} in the pipeline.
    * @deprecated use {@link #DumpPipe(String, PrintStream, Pipe)} instead
    */
   public DumpPipe(PrintStream out, Pipe next)
   {
      this("DumpPipe",out,next);
   }

   /**
    * Copy constructor.
    */
   private DumpPipe(DumpPipe that, PipeCloner cloner)
   {
      super(that,cloner);
      this.name = that.name;
      this.out = that.out;
      this.staxOut = that.staxOut;
   }

   public Packet process(Packet packet)
   {
      dump("request",packet);
      Packet reply = next.process(packet);
      dump("response",reply);
      return reply;
   }

   private void dump(String header, Packet packet)
   {
      if(!msgLog.isTraceEnabled())
         return;

      msgLog.trace("====[" + name + ":" + header + "]====");
      if(packet.getMessage() == null)
      {
         msgLog.trace("(none)");
      }
      else
      {
         ByteArrayOutputStream bout = new ByteArrayOutputStream();
         try
         {
            XMLStreamWriter writer = staxOut.createXMLStreamWriter(new PrintStream(bout)
            {
               public void close()
               {
                  // does nothing
               }
            });
            writer = createIndenter(writer);
            packet.getMessage().copy().writeTo(writer);
            writer.close();
         }
         catch (XMLStreamException e)
         {
            e.printStackTrace(new PrintStream(bout));
         }
         
         msgLog.trace(new String(bout.toByteArray()));
      }
      msgLog.trace("============");
   }

   /**
    * Wraps {@link XMLStreamWriter} by an indentation engine if available.
    * <p>
    * We can do this only if we have <tt>stax-utils.jar</tt> on the classpath.
    * </p>
    */
   @SuppressWarnings("unchecked")
   private XMLStreamWriter createIndenter(XMLStreamWriter writer)
   {
      try
      {
         Class clazz = getClass().getClassLoader().loadClass("javanet.staxutils.IndentingXMLStreamWriter");
         Constructor c = clazz.getConstructor(XMLStreamWriter.class);
         writer = (XMLStreamWriter)c.newInstance(writer);
      }
      catch (Exception e)
      {
         msgLog.warn("WARNING: put stax-utils.jar to the classpath to indent dumped output");
      }
      return writer;
   }

   public Pipe copy(PipeCloner cloner)
   {
      return new DumpPipe(this, cloner);
   }

   public void preDestroy()
   {
      // does nothing
   }

}
