/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.stack.metro.metadata;

import com.sun.xml.ws.api.server.BoundEndpoint;
import com.sun.xml.ws.api.server.Container;
import com.sun.xml.ws.api.server.Module;
import com.sun.xml.ws.api.server.ResourceInjector;
import com.sun.xml.ws.api.server.ServerPipelineHook;
import com.sun.xml.ws.transport.http.DeploymentDescriptorParser;
import com.sun.xml.ws.transport.http.ResourceLoader;
import com.sun.xml.ws.transport.http.servlet.ServletAdapter;
import com.sun.xml.ws.transport.http.servlet.ServletAdapterList;

import org.jboss.logging.Logger;
import org.jboss.wsf.common.ResourceLoaderAdapter;
import org.jboss.wsf.common.integration.WSConstants;
import org.jboss.wsf.spi.deployment.ArchiveDeployment;
import org.jboss.wsf.spi.deployment.Deployment;
import org.jboss.wsf.spi.deployment.DeploymentAspect;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.deployment.UnifiedVirtualFile;
import org.jboss.wsf.stack.metro.DeploymentDescriptorParserExt;
import org.jboss.wsf.stack.metro.DeploymentDescriptorParserJBWS;
import org.jboss.wsf.stack.metro.MessageStreamAdapter;
import org.jboss.wsf.stack.metro.log.MessageLogPipelineHook;

import javax.xml.ws.WebServiceException;
import javax.servlet.ServletContext;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Map;
import java.util.List;

/**
 * Creates a Metro runtime model and associates it with the deployment.
 * Doing so requires a valid runtime loader present, that will act as a temporary
 * thread local classloader. <p>The <code>spi.Endpoint</code> carries a Metro <code>ServletAdapter</code>
 * that is associated with the runtime model:<br>
 * <code>(Endpoint.getAttachment(ServletAdapter.class))</code>
 * This adapter acts as the main entry point for SOAP/HTTP requests.
 * Please note that the current implementation doesnt support the stream based RequestHandler API.
 *
 * @see org.jboss.wsf.stack.metro.RequestHandlerImpl
 * @see org.jboss.wsf.stack.metro.DeploymentDescriptorParserExt
 *
 * @author Heiko.Braun@jboss.com
 */
public class RuntimeModelDeploymentAspect extends DeploymentAspect
{
   private static final org.jboss.logging.Logger log = Logger.getLogger(RuntimeModelDeploymentAspect.class);
   protected static final String JAXWS_RI_RUNTIME = "/WEB-INF/sun-jaxws.xml";
   public static final String PARAM_SUN_JAXWS_URL = "jbossws-sun-jaxws-url";
   
   public void start(Deployment deployment)
   {
      ClassLoader runtimeLoader = deployment.getRuntimeClassLoader();
      if(null == runtimeLoader)
         throw new IllegalArgumentException("Null runtimeLoader");

      ClassLoader origClassLoader = Thread.currentThread().getContextClassLoader();
      try
      {
         // Metro uses the thread context classloader in several places
         Thread.currentThread().setContextClassLoader(runtimeLoader);

         UnifiedVirtualFile vfsRoot = null;
         if (deployment instanceof ArchiveDeployment)
         {
            vfsRoot = ((ArchiveDeployment)deployment).getRootFile();
         }
         else
         {                        
            vfsRoot = new ResourceLoaderAdapter(runtimeLoader);
         }

         ResourceLoader resourceLoader = new VFSResourceLoader(vfsRoot);
         JBossWSContainer container = new JBossWSContainer(resourceLoader);

         // TODO: refactor to general HTTPAdapter usage 
         ServletAdapterList adapterList = new ServletAdapterList();

         // Parse the descriptor file and build endpoint infos
         DeploymentDescriptorParserExt<ServletAdapter> parser =
           createDeploymentDescriptorParser(runtimeLoader, container, resourceLoader, adapterList, vfsRoot);
         
         URL sunJaxWsXml = getDDUrlFromContext(deployment);
         
         List<ServletAdapter> adapters = parser.parse(sunJaxWsXml.toExternalForm(), sunJaxWsXml.openStream());
         
         for(ServletAdapter adapter : adapters)
         {            
            for(Endpoint ep : deployment.getService().getEndpoints())
            {
               // TODO matching need to be improved
               boolean nameEquals = ep.getShortName().equals(adapter.getName());
               boolean beanEquals = ep.getTargetBeanClass().equals(adapter.getEndpoint().getImplementationClass()); 
               if(nameEquals && beanEquals)
               {
                  MessageStreamAdapter msgStreamAdapter = MessageStreamAdapter.FACTORY.createAdapter(adapter.getName(), null, adapter.getEndpoint()); 
                  // TODO JBWS-1869: Support stream based RequestHandler API 
                  if (ep.getAttachment(MessageStreamAdapter.class) == null) // TODO: review to don't insert the same attachments multiple times 
                     ep.addAttachment(MessageStreamAdapter.class, msgStreamAdapter); // do it only once
                  if (ep.getAttachment(ServletAdapter.class) == null)
                     ep.addAttachment(ServletAdapter.class, adapter); // TODO: review to don't insert the same attachments multiple times
                  break;
               }
            }                       
         }
      }
      catch (Throwable e)
      {         
         throw new WebServiceException("Failed to build METRO runtime model", e);
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(origClassLoader);
      }
   }

   @SuppressWarnings("unchecked")
   private URL getDDUrlFromContext(Deployment deployment) throws MalformedURLException
   {
      Map<String, String> contextProperties =
        (Map<String, String>)deployment.getProperty(WSConstants.STACK_CONTEXT_PARAMS);

      return new URL(contextProperties.get(PARAM_SUN_JAXWS_URL));
   }

   /**
    * Externalized for integration
    */
   protected DeploymentDescriptorParserExt<ServletAdapter> createDeploymentDescriptorParser 
   (
      ClassLoader classLoader,
      JBossWSContainer container,
      ResourceLoader resourceLoader,
      DeploymentDescriptorParser.AdapterFactory<ServletAdapter> adapterList,
      UnifiedVirtualFile vfsRoot
   )
   throws MalformedURLException
   {
      DeploymentDescriptorParserJBWS<ServletAdapter> parser = new DeploymentDescriptorParserJBWS<ServletAdapter>(classLoader, resourceLoader, container, adapterList);
      parser.setArchiveFile(vfsRoot);
      return parser;
   }

   /**
    * Externalized for integration
    */
   protected URL getDDUrlFromResourceLoader(UnifiedVirtualFile vfsRoot) throws IOException
   {
      return vfsRoot.findChild(JAXWS_RI_RUNTIME).toURL();
   }

   /**
    * Provides access to {@link javax.servlet.ServletContext} via {@link com.sun.xml.ws.api.server.Container}. 
    * Pipes can get ServletContext from Container and use it to load some resources.
    */
   private static class JBossWSContainer extends Container
   {
      private ResourceLoader resourceLoader;
      private Module module = new Module()
      {
         private final List<BoundEndpoint> endpoints = new ArrayList<BoundEndpoint>();

         public List<BoundEndpoint> getBoundEndpoints()
         {
            return endpoints;
         }
      };

      JBossWSContainer(ResourceLoader loader)
      {
         this.resourceLoader = loader;
      }

      @SuppressWarnings("unchecked")
      public <T> T getSPI(Class<T> spiType)
      {
         if (spiType == Module.class)
         {
            return spiType.cast(module);
         }
         if(spiType == ResourceInjector.class)
         {
            return (T)ResourceInjector.STANDALONE;
         }
         if (spiType == ServerPipelineHook.class)
         {
            return (T)new MessageLogPipelineHook();
         }
         if(spiType == ServletContext.class)
         {
            return (T)new FakeServletContext(resourceLoader);
         }

         log.warn("Unable to resolve SPI for type: " + spiType);
         return null;
      }
   }

}
