/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.xnio.helpers;

import org.jboss.xnio.ConfigurableFactory;
import org.jboss.xnio.IoUtils;
import org.jboss.xnio.Xnio;
import org.jboss.xnio.IoHandlerFactory;
import org.jboss.xnio.CloseableTcpConnector;
import org.jboss.xnio.log.Logger;
import org.jboss.xnio.channels.CommonOptions;
import org.jboss.xnio.channels.ChannelOption;
import org.jboss.xnio.channels.TcpChannel;
import org.jboss.xnio.channels.BoundServer;
import org.jboss.xnio.channels.BoundChannel;
import org.jboss.xnio.channels.UdpChannel;
import java.io.Closeable;
import java.io.IOException;
import java.net.SocketAddress;
import java.util.concurrent.Executor;

/**
 *
 */
public final class XnioController<T extends Closeable> {

    private static final Logger log = Logger.getLogger("org.jboss.xnio.deployer");

    private final ConfigurableFactory<? extends T> configurableFactory;
    private T handle;

    private XnioController(final ConfigurableFactory<? extends T> factory) {
        configurableFactory = factory;
    }

    public void stop() {
        IoUtils.safeClose(handle);
        handle = null;
    }

    public void start() throws IOException {
        handle = configurableFactory.create();
    }

    public T getHandle() {
        return handle;
    }

    private <E> void doSetOption(final ChannelOption<E> option, final E value) throws IOException {
        final ConfigurableFactory<? extends T> factory = configurableFactory;
        if (value != null) {
            if (factory.getOptions().contains(option)) {
                factory.setOption(option, value);
            } else {
                log.debug("XNIO component does not support option \"%s\"", option);
            }
        }
    }

    public void setBacklog(Integer backlog) throws IOException {
        doSetOption(CommonOptions.BACKLOG, backlog);
    }

    public void setBroadcast(Boolean broadcast) throws IOException {
        doSetOption(CommonOptions.BROADCAST, broadcast);
    }

    public void setCloseAbort(Boolean closeAbort) throws IOException {
        doSetOption(CommonOptions.CLOSE_ABORT, closeAbort);
    }

    public void setIpTrafficClass(Integer ipTrafficClass) throws IOException {
        doSetOption(CommonOptions.IP_TRAFFIC_CLASS, ipTrafficClass);
    }

    public void setKeepAlive(Boolean keepAlive) throws IOException {
        doSetOption(CommonOptions.KEEP_ALIVE, keepAlive);
    }

    public void setManageConnections(Boolean manageConnections) throws IOException {
        doSetOption(CommonOptions.MANAGE_CONNECTIONS, manageConnections);
    }

    public void setMulticastTtl(Integer multicastTtl) throws IOException {
        doSetOption(CommonOptions.MULTICAST_TTL, multicastTtl);
    }

    public void setOobInline(Boolean oobInline) throws IOException {
        doSetOption(CommonOptions.TCP_OOB_INLINE, oobInline);
    }

    public void setReceiveBufferSize(Integer receiveBufferSize) throws IOException {
        doSetOption(CommonOptions.RECEIVE_BUFFER, receiveBufferSize);
    }

    public void setReuseAddresses(Boolean reuseAddresses) throws IOException {
        doSetOption(CommonOptions.REUSE_ADDRESSES, reuseAddresses);
    }

    public void setSendBufferSize(Integer sendBufferSize) throws IOException {
        doSetOption(CommonOptions.SEND_BUFFER, sendBufferSize);
    }

    public void setTcpNoDelay(Boolean tcpNoDelay) throws IOException {
        doSetOption(CommonOptions.TCP_NODELAY, tcpNoDelay);
    }

    private static <X extends Closeable> XnioController<X> create(final ConfigurableFactory<X> factory) {
        return new XnioController<X>(factory);
    }

    public static XnioController<BoundServer<SocketAddress, BoundChannel<SocketAddress>>> tcpServer(final Xnio xnio, final IoHandlerFactory<? super TcpChannel> handlerFactory, final SocketAddress[] bindAddresses) {
        return create(xnio.createTcpServer(handlerFactory, bindAddresses));
    }

    public static XnioController<BoundServer<SocketAddress, BoundChannel<SocketAddress>>> tcpServer(final Xnio xnio, final Executor executor, final IoHandlerFactory<? super TcpChannel> handlerFactory, final SocketAddress[] bindAddresses) {
        return create(xnio.createTcpServer(executor, handlerFactory, bindAddresses));
    }

    public static XnioController<BoundServer<SocketAddress, UdpChannel>> udpServer(final Xnio xnio, final boolean multicast, final IoHandlerFactory<? super UdpChannel> handlerFactory, final SocketAddress[] bindAddresses) {
        return create(xnio.createUdpServer(multicast, handlerFactory, bindAddresses));
    }

    public static XnioController<BoundServer<SocketAddress, UdpChannel>> udpServer(final Xnio xnio, final Executor executor, final boolean multicast, final IoHandlerFactory<? super UdpChannel> handlerFactory, final SocketAddress[] bindAddresses) {
        return create(xnio.createUdpServer(executor, multicast, handlerFactory, bindAddresses));
    }

    public static XnioController<CloseableTcpConnector> tcpConnector(final Xnio xnio) {
        return create(xnio.createTcpConnector());
    }

    public static XnioController<CloseableTcpConnector> tcpConnector(final Xnio xnio, final Executor executor) {
        return create(xnio.createTcpConnector(executor));
    }
}
