/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.xnio.nio;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.DatagramChannel;
import java.nio.channels.Pipe;
import java.nio.channels.SelectionKey;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import java.util.concurrent.locks.LockSupport;
import org.xnio.Bits;
import org.xnio.Cancellable;
import org.xnio.ChannelListener;
import org.xnio.ChannelListeners;
import org.xnio.ChannelPipe;
import org.xnio.ClosedWorkerException;
import org.xnio.FailedIoFuture;
import org.xnio.FinishedIoFuture;
import org.xnio.FutureResult;
import org.xnio.IoFuture;
import org.xnio.IoUtils;
import org.xnio.Option;
import org.xnio.OptionMap;
import org.xnio.Options;
import org.xnio.XnioWorker;
import org.xnio.channels.AcceptingChannel;
import org.xnio.channels.BoundChannel;
import org.xnio.channels.CloseableChannel;
import org.xnio.channels.ConnectedStreamChannel;
import org.xnio.channels.MulticastMessageChannel;
import org.xnio.channels.StreamChannel;
import org.xnio.channels.StreamSinkChannel;
import org.xnio.channels.StreamSourceChannel;

import static org.xnio.IoUtils.safeClose;
import static org.xnio.ChannelListener.SimpleSetter;
import static org.xnio.nio.Log.log;

/**
 * @author <a href="mailto:david.lloyd@redhat.com">David M. Lloyd</a>
 */
final class NioXnioWorker extends XnioWorker {

    private static final int CLOSE_REQ = (1 << 31);
    private static final int CLOSE_COMP = (1 << 30);

    // start at 1 for the provided thread pool
    private volatile int state = 1;

    private final WorkerThread[] readWorkers;
    private final WorkerThread[] writeWorkers;

    @SuppressWarnings("unused")
    private volatile Thread shutdownWaiter;

    private static final AtomicReferenceFieldUpdater<NioXnioWorker, Thread> shutdownWaiterUpdater = AtomicReferenceFieldUpdater.newUpdater(NioXnioWorker.class, Thread.class, "shutdownWaiter");

    private static final AtomicIntegerFieldUpdater<NioXnioWorker> stateUpdater = AtomicIntegerFieldUpdater.newUpdater(NioXnioWorker.class, "state");

    NioXnioWorker(final NioXnio xnio, final ThreadGroup threadGroup, final OptionMap optionMap, final Runnable terminationTask) throws IOException {
        super(xnio, threadGroup, optionMap, terminationTask);
        final int readCount = optionMap.get(Options.WORKER_READ_THREADS, 1);
        if (readCount < 0) {
            throw new IllegalArgumentException("Worker read thread count must be >= 0");
        }
        final int writeCount = optionMap.get(Options.WORKER_WRITE_THREADS, 1);
        if (writeCount < 0) {
            throw new IllegalArgumentException("Worker write thread count must be >= 0");
        }
        final long workerStackSize = optionMap.get(Options.STACK_SIZE, 0L);
        if (workerStackSize < 0L) {
            throw new IllegalArgumentException("Worker stack size must be >= 0");
        }
        String workerName = getName();
        WorkerThread[] readWorkers, writeWorkers;
        readWorkers = new WorkerThread[readCount];
        writeWorkers = new WorkerThread[writeCount];
        final boolean markWorkerThreadAsDaemon = optionMap.get(Options.THREAD_DAEMON, false);
        boolean ok = false;
        try {
            for (int i = 0; i < readCount; i++) {
                final WorkerThread readWorker = new WorkerThread(this, xnio.mainSelectorCreator.open(), String.format("%s read-%d", workerName, Integer.valueOf(i + 1)), threadGroup, workerStackSize, false);
                // Mark as daemon if the Options.THREAD_DAEMON has been set
                if (markWorkerThreadAsDaemon) {
                    readWorker.setDaemon(true);
                }
                readWorkers[i] = readWorker;
            }
            for (int i = 0; i < writeCount; i++) {
                final WorkerThread writeWorker = new WorkerThread(this, xnio.mainSelectorCreator.open(), String.format("%s write-%d", workerName, Integer.valueOf(i + 1)), threadGroup, workerStackSize, true);
                // Mark as daemon if Options.THREAD_DAEMON has been set
                if (markWorkerThreadAsDaemon) {
                    writeWorker.setDaemon(true);
                }
                writeWorkers[i] = writeWorker;
            }
            ok = true;
        } finally {
            if (! ok) {
                for (WorkerThread worker : readWorkers) {
                    if (worker != null) safeClose(worker.getSelector());
                }
                for (WorkerThread worker : writeWorkers) {
                    if (worker != null) safeClose(worker.getSelector());
                }
            }
        }
        this.readWorkers = readWorkers;
        this.writeWorkers = writeWorkers;
    }

    void start() {
        for (WorkerThread worker : readWorkers) {
            openResourceUnconditionally();
            worker.start();
        }
        for (WorkerThread worker : writeWorkers) {
            openResourceUnconditionally();
            worker.start();
        }
    }

    private static final WorkerThread[] NO_WORKERS = new WorkerThread[0];

    WorkerThread chooseOptional(final boolean write) {
        final WorkerThread[] orig = write ? writeWorkers : readWorkers;
        final int length = orig.length;
        if (length == 0) {
            return null;
        }
        if (length == 1) {
            return orig[0];
        }
        final Random random = IoUtils.getThreadLocalRandom();
        return orig[random.nextInt(length)];
    }

    WorkerThread choose(final boolean write) {
        final WorkerThread result = chooseOptional(write);
        if (result == null) {
            throw new IllegalArgumentException("No threads configured");
        }
        return result;
    }

    WorkerThread[] choose(int count, boolean write) {
        if (count == 0) {
            return NO_WORKERS;
        }
        final WorkerThread[] orig = write ? writeWorkers : readWorkers;
        final int length = orig.length;
        final int halfLength = length >> 1;
        if (length == 0) {
            throw new IllegalArgumentException("No threads configured");
        }
        if (count == length) {
            return orig;
        }
        if (count > length) {
            throw new IllegalArgumentException("Not enough " + (write ? "write" : "read") + " threads configured");
        }
        final WorkerThread[] result = new WorkerThread[count];
        final Random random = IoUtils.getThreadLocalRandom();
        if (count == 1) {
            result[0] = orig[random.nextInt(length)];
            return result;
        }
        if (length < 32) {
            if (count >= halfLength) {
                int bits = (1 << length) - 1;
                do {
                    bits &= ~(1 << random.nextInt(length));
                } while (Integer.bitCount(bits) > count);
                for (int i = 0; i < count; i ++) {
                    final int bit = Integer.numberOfTrailingZeros(bits);
                    result[i] = orig[bit];
                    bits ^= Integer.lowestOneBit(bits);
                }
                return result;
            } else {
                int bits = 0;
                do {
                    bits |= (1 << random.nextInt(length));
                } while (Integer.bitCount(bits) < count);
                for (int i = 0; i < count; i ++) {
                    final int bit = Integer.numberOfTrailingZeros(bits);
                    result[i] = orig[bit];
                    bits ^= Integer.lowestOneBit(bits);
                }
                return result;
            }
        }
        if (length < 64) {
            if (count >= halfLength) {
                long bits = (1L << (long) length) - 1L;
                do {
                    bits &= ~(1L << (long) random.nextInt(length));
                } while (Long.bitCount(bits) > count);
                for (int i = 0; i < count; i ++) {
                    final int bit = Long.numberOfTrailingZeros(bits);
                    result[i] = orig[bit];
                    bits ^= Long.lowestOneBit(bits);
                }
                return result;
            } else {
                long bits = 0;
                do {
                    bits |= (1L << (long) random.nextInt(length));
                } while (Long.bitCount(bits) < count);
                for (int i = 0; i < count; i ++) {
                    final int bit = Long.numberOfTrailingZeros(bits);
                    result[i] = orig[bit];
                    bits ^= Long.lowestOneBit(bits);
                }
                return result;
            }
        }
        // lots of threads.  No faster way to do it.
        final HashSet<WorkerThread> set;
        if (count >= halfLength) {
            // We're returning half or more of the threads.
            set = new HashSet<WorkerThread>(Arrays.asList(orig));
            while (set.size() > count) {
                set.remove(orig[random.nextInt(length)]);
            }
        } else {
            // We're returning less than half of the threads.
            set = new HashSet<WorkerThread>(length);
            while (set.size() < count) {
                set.add(orig[random.nextInt(length)]);
            }
        }
        return set.toArray(result);
    }

    protected AcceptingChannel<? extends ConnectedStreamChannel> createTcpServer(final InetSocketAddress bindAddress, final ChannelListener<? super AcceptingChannel<ConnectedStreamChannel>> acceptListener, final OptionMap optionMap) throws IOException {
        checkShutdown();
        boolean ok = false;
        final ServerSocketChannel channel = ServerSocketChannel.open();
        try {
            channel.configureBlocking(false);
            if (optionMap.contains(Options.BACKLOG)) {
                channel.socket().bind(bindAddress, optionMap.get(Options.BACKLOG, 128));
            } else {
                channel.socket().bind(bindAddress);
            }
            final NioTcpServer server = new NioTcpServer(this, channel, optionMap);
            final ChannelListener.SimpleSetter<NioTcpServer> setter = server.getAcceptSetter();
            // not unsafe - http://youtrack.jetbrains.net/issue/IDEA-59290
            //noinspection unchecked
            setter.set((ChannelListener<? super NioTcpServer>) acceptListener);
            ok = true;
            return server;
        } finally {
            if (! ok) {
                IoUtils.safeClose(channel);
            }
        }
    }

    protected IoFuture<ConnectedStreamChannel> connectTcpStream(final InetSocketAddress bindAddress, final InetSocketAddress destinationAddress, final ChannelListener<? super ConnectedStreamChannel> openListener, final ChannelListener<? super BoundChannel> bindListener, final OptionMap optionMap) {
        try {
            checkShutdown();
        } catch (ClosedWorkerException e) {
            return new FailedIoFuture<ConnectedStreamChannel>(e);
        }
        try {
            final SocketChannel channel = SocketChannel.open();
            channel.configureBlocking(false);
            channel.socket().bind(bindAddress);
            final NioTcpChannel tcpChannel = new NioTcpChannel(this, null, channel);
            tcpChannel.start();
            final NioHandle<NioTcpChannel> connectHandle = optionMap.get(Options.WORKER_ESTABLISH_WRITING, false) ? tcpChannel.getWriteHandle() : tcpChannel.getReadHandle();
            final int oldOps = connectHandle.setOps(SelectionKey.OP_CONNECT);
            if (connectHandle == null) {
                throw new IllegalArgumentException("Wrong value for option " + Options. WORKER_ESTABLISH_WRITING +
                        ". This NioWorker has no " + (optionMap.get(Options.WORKER_ESTABLISH_WRITING, false)? "write": "read")
                        + " thread.");
            }
            ChannelListeners.invokeChannelListener(tcpChannel.getBoundChannel(), bindListener);
            if (channel.connect(destinationAddress)) {
                // not unsafe - http://youtrack.jetbrains.net/issue/IDEA-59290
                //noinspection unchecked
                connectHandle.getWorkerThread().execute(ChannelListeners.getChannelListenerTask(tcpChannel, openListener));
                return new FinishedIoFuture<ConnectedStreamChannel>(tcpChannel);
            }
            final SimpleSetter<NioTcpChannel> setter = connectHandle.getHandlerSetter();
            final FutureResult<ConnectedStreamChannel> futureResult = new FutureResult<ConnectedStreamChannel>();
            setter.set(new ChannelListener<NioTcpChannel>() {
                public void handleEvent(final NioTcpChannel channel) {
                    final SocketChannel socketChannel = channel.getReadChannel();
                    try {
                        if (socketChannel.finishConnect()) {
                            connectHandle.suspend();
                            connectHandle.getHandlerSetter().set(null);
                            connectHandle.setOps(oldOps);
                            if (!futureResult.setResult(tcpChannel)) {
                                // if futureResult is canceled, close channel
                                IoUtils.safeClose(channel);
                            } else {
                                channel.configureFrom(optionMap);
                                //noinspection unchecked
                                ChannelListeners.invokeChannelListener(tcpChannel, openListener);
                            }
                        }
                    } catch (IOException e) {
                        IoUtils.safeClose(channel);
                        futureResult.setException(e);
                    }
                }

                public String toString() {
                    return "Connection finisher for " + channel;
                }
            });
            futureResult.addCancelHandler(new Cancellable() {
                public Cancellable cancel() {
                    if (futureResult.setCancelled()) {
                        IoUtils.safeClose(tcpChannel);
                    }
                    return this;
                }

                public String toString() {
                    return "Cancel handler for " + channel;
                }
            });
            connectHandle.resume();
            return futureResult.getIoFuture();
        } catch (IOException e) {
            return new FailedIoFuture<ConnectedStreamChannel>(e);
        }
    }

    protected IoFuture<ConnectedStreamChannel> acceptTcpStream(final InetSocketAddress destination, final ChannelListener<? super ConnectedStreamChannel> openListener, final ChannelListener<? super BoundChannel> bindListener, final OptionMap optionMap) {
        try {
            checkShutdown();
        } catch (ClosedWorkerException e) {
            return new FailedIoFuture<ConnectedStreamChannel>(e);
        }
        final WorkerThread connectThread = choose(optionMap.get(Options.WORKER_ESTABLISH_WRITING, false));
        try {
            final ServerSocketChannel channel = ServerSocketChannel.open();
            channel.configureBlocking(false);
            channel.socket().bind(destination);
            final ChannelListener.SimpleSetter<NioTcpChannel> closeSetter = new ChannelListener.SimpleSetter<NioTcpChannel>();
            //noinspection unchecked
            ChannelListeners.invokeChannelListener(new BoundChannel() {
                public XnioWorker getWorker() {
                    return NioXnioWorker.this;
                }

                public SocketAddress getLocalAddress() {
                    return channel.socket().getLocalSocketAddress();
                }

                public <A extends SocketAddress> A getLocalAddress(final Class<A> type) {
                    final SocketAddress address = getLocalAddress();
                    return type.isInstance(address) ? type.cast(address) : null;
                }

                public ChannelListener.Setter<? extends BoundChannel> getCloseSetter() {
                    return closeSetter;
                }

                public boolean isOpen() {
                    return channel.isOpen();
                }

                public boolean supportsOption(final Option<?> option) {
                    return false;
                }

                public <T> T getOption(final Option<T> option) throws IOException {
                    return null;
                }

                public <T> T setOption(final Option<T> option, final T value) throws IllegalArgumentException, IOException {
                    return null;
                }

                public void close() throws IOException {
                    channel.close();
                }

                public String toString() {
                    return String.format("TCP acceptor bound channel (NIO) <%h>", this);
                }
            }, bindListener);
            final SocketChannel accepted = channel.accept();
            if (accepted != null) {
                IoUtils.safeClose(channel);
                final NioTcpChannel tcpChannel = new NioTcpChannel(this, null, accepted);
                tcpChannel.start();
                tcpChannel.configureFrom(optionMap);
                //noinspection unchecked
                ChannelListeners.invokeChannelListener(tcpChannel, openListener);
                return new FinishedIoFuture<ConnectedStreamChannel>(tcpChannel);
            }
            final SimpleSetter<ServerSocketChannel> setter = new SimpleSetter<ServerSocketChannel>();
            final FutureResult<ConnectedStreamChannel> futureResult = new FutureResult<ConnectedStreamChannel>();
            final NioHandle<ServerSocketChannel> handle = connectThread.addChannel(channel, channel, SelectionKey.OP_ACCEPT, setter);
            setter.set(new ChannelListener<ServerSocketChannel>() {
                public void handleEvent(final ServerSocketChannel channel) {
                    final SocketChannel accepted;
                    try {
                        accepted = channel.accept();
                        if (accepted == null) {
                            return;
                        }
                    } catch (IOException e) {
                        IoUtils.safeClose(channel);
                        handle.cancelKey();
                        futureResult.setException(e);
                        return;
                    }
                    boolean ok = false;
                    try {
                        handle.cancelKey();
                        IoUtils.safeClose(channel);
                        try {
                            accepted.configureBlocking(false);
                            final NioTcpChannel tcpChannel;
                            tcpChannel = new NioTcpChannel(NioXnioWorker.this, null, accepted);
                            tcpChannel.start();
                            tcpChannel.configureFrom(optionMap);
                            futureResult.setResult(tcpChannel);
                            ok = true;
                            //noinspection unchecked
                            ChannelListeners.invokeChannelListener(tcpChannel, openListener);
                        } catch (IOException e) {
                            futureResult.setException(e);
                            return;
                        }
                    } finally {
                        if (! ok) {
                            IoUtils.safeClose(accepted);
                        }
                    }
                }

                public String toString() {
                    return "Accepting finisher for " + channel;
                }
            });
            handle.resume();
            return futureResult.getIoFuture();
        } catch (IOException e) {
            return new FailedIoFuture<ConnectedStreamChannel>(e);
        }
    }

    /** {@inheritDoc} */
    public MulticastMessageChannel createUdpServer(final InetSocketAddress bindAddress, final ChannelListener<? super MulticastMessageChannel> bindListener, final OptionMap optionMap) throws IOException {
        checkShutdown();
        final DatagramChannel channel = DatagramChannel.open();
        channel.configureBlocking(false);
        channel.socket().bind(bindAddress);
        final NioUdpChannel udpChannel = new NioUdpChannel(this, channel);
        udpChannel.start();
        //noinspection unchecked
        ChannelListeners.invokeChannelListener(udpChannel, bindListener);
        return udpChannel;
    }

    public ChannelPipe<StreamChannel, StreamChannel> createFullDuplexPipe() throws IOException {
        checkShutdown();
        boolean ok = false;
        final Pipe in = Pipe.open();
        try {
            in.source().configureBlocking(false);
            in.sink().configureBlocking(false);
            final Pipe out = Pipe.open();
            try {
                out.source().configureBlocking(false);
                out.sink().configureBlocking(false);
                final NioPipeChannel left = new NioPipeChannel(NioXnioWorker.this, in.sink(), out.source());
                left.start();
                final NioPipeChannel right = new NioPipeChannel(NioXnioWorker.this, out.sink(), in.source());
                right.start();
                final ChannelPipe<StreamChannel, StreamChannel> result = new ChannelPipe<StreamChannel, StreamChannel>(left, right);
                ok = true;
                return result;
            } finally {
                if (! ok) {
                    safeClose(out.sink());
                    safeClose(out.source());
                }
            }
        } finally {
            if (! ok) {
                safeClose(in.sink());
                safeClose(in.source());
            }
        }
    }

    public ChannelPipe<StreamSourceChannel, StreamSinkChannel> createHalfDuplexPipe() throws IOException {
        checkShutdown();
        final Pipe pipe = Pipe.open();
        boolean ok = false;
        try {
            pipe.source().configureBlocking(false);
            pipe.sink().configureBlocking(false);
            final NioPipeSourceChannel sourceChannel = new NioPipeSourceChannel(this, pipe.source());
            sourceChannel.start();
            final NioPipeSinkChannel sinkChannel = new NioPipeSinkChannel(this, pipe.sink());
            sinkChannel.start();
            final ChannelPipe<StreamSourceChannel,StreamSinkChannel> result = new ChannelPipe<StreamSourceChannel, StreamSinkChannel>(sourceChannel, sinkChannel);
            ok = true;
            return result;
        } finally {
            if (! ok) {
                safeClose(pipe.sink());
                safeClose(pipe.source());
            }
        }
    }

    public boolean isShutdown() {
        return (state & CLOSE_REQ) != 0;
    }

    public boolean isTerminated() {
        return (state & CLOSE_COMP) != 0;
    }

    /**
     * Open a resource unconditionally (i.e. accepting a connection on an open server).
     */
    void openResourceUnconditionally() {
        int oldState = stateUpdater.getAndIncrement(this);
        if (log.isTraceEnabled()) {
            log.tracef("CAS %s %08x -> %08x", this, Integer.valueOf(oldState), Integer.valueOf(oldState + 1));
        }
    }

    void checkShutdown() throws ClosedWorkerException {
        if (isShutdown()) throw new ClosedWorkerException("Worker is shut down");
    }

    void closeResource() {
        int oldState = stateUpdater.decrementAndGet(this);
        if (log.isTraceEnabled()) {
            log.tracef("CAS %s %08x -> %08x", this, Integer.valueOf(oldState + 1), Integer.valueOf(oldState));
        }
        while (oldState == CLOSE_REQ) {
            if (stateUpdater.compareAndSet(this, CLOSE_REQ, CLOSE_REQ | CLOSE_COMP)) {
                log.tracef("CAS %s %08x -> %08x (close complete)", this, Integer.valueOf(CLOSE_REQ), Integer.valueOf(CLOSE_REQ | CLOSE_COMP));
                safeUnpark(shutdownWaiterUpdater.getAndSet(this, null));
                final Runnable task = getTerminationTask();
                if (task != null) try {
                    task.run();
                } catch (Throwable ignored) {}
                return;
            }
            oldState = state;
        }
    }

    public void shutdown() {
        int oldState;
        oldState = state;
        while ((oldState & CLOSE_REQ) == 0) {
            // need to do the close ourselves...
            if (! stateUpdater.compareAndSet(this, oldState, oldState | CLOSE_REQ)) {
                // changed in the meantime
                oldState = state;
                continue;
            }
            log.tracef("Initiating shutdown of %s", this);
            for (WorkerThread worker : readWorkers) {
                worker.shutdown();
            }
            for (WorkerThread worker : writeWorkers) {
                worker.shutdown();
            }
            shutDownTaskPool();
            return;
        }
        log.tracef("Idempotent shutdown of %s", this);
        return;
    }

    public List<Runnable> shutdownNow() {
        shutdown();
        return shutDownTaskPoolNow();
    }

    public boolean awaitTermination(final long timeout, final TimeUnit unit) throws InterruptedException {
        int oldState = state;
        if (Bits.allAreSet(oldState, CLOSE_COMP)) {
            return true;
        }
        long then = System.nanoTime();
        long duration = unit.toNanos(timeout);
        final Thread myThread = Thread.currentThread();
        while (Bits.allAreClear(oldState = state, CLOSE_COMP)) {
            final Thread oldThread = shutdownWaiterUpdater.getAndSet(this, myThread);
            try {
                if (Bits.allAreSet(oldState = state, CLOSE_COMP)) {
                    break;
                }
                LockSupport.parkNanos(this, duration);
                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }
                long now = System.nanoTime();
                duration -= now - then;
                if (duration < 0L) {
                    oldState = state;
                    break;
                }
            } finally {
                safeUnpark(oldThread);
            }
        }
        return Bits.allAreSet(oldState, CLOSE_COMP);
    }

    public void awaitTermination() throws InterruptedException {
        int oldState = state;
        if (Bits.allAreSet(oldState, CLOSE_COMP)) {
            return;
        }
        final Thread myThread = Thread.currentThread();
        while (Bits.allAreClear(state, CLOSE_COMP)) {
            final Thread oldThread = shutdownWaiterUpdater.getAndSet(this, myThread);
            try {
                if (Bits.allAreSet(state, CLOSE_COMP)) {
                    break;
                }
                LockSupport.park(this);
                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }
            } finally {
                safeUnpark(oldThread);
            }
        }
    }

    private static void safeUnpark(final Thread waiter) {
        if (waiter != null) LockSupport.unpark(waiter);
    }

    protected void doMigration(final CloseableChannel channel) throws ClosedChannelException {
        if (channel.getWorker() == this) {
            return;
        }
        ((AbstractNioChannel<?>)channel).migrateTo(this);
    }

    protected void taskPoolTerminated() {
        closeResource();
    }

    public NioXnio getXnio() {
        return (NioXnio) super.getXnio();
    }
}
