/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.jboss.bpm.console.client.model.ProcessDefinitionWrapper;
import org.jboss.bpm.console.client.model.ProcessInstanceWrapper;
import org.jboss.bpm.console.client.model.ProcessInstance;
import org.jboss.bpm.console.server.dao.DAOFactory;
import org.jboss.bpm.console.server.dao.ProcessDAO;
import org.jboss.bpm.console.server.gson.GsonFactory;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

/**
 * REST server module for accessing process related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("process")
public class ProcessManagement
{
   private ProcessDAO processDAO;

   /**
    * Lazy load the {@link org.jboss.bpm.console.server.dao.ProcessDAO}   
    */
   private ProcessDAO getProcessDAO(HttpServletRequest request)
   {
      if(null==this.processDAO)
      {
         DAOFactory factory = DAOFactory.newInstance(request.getSession().getServletContext());
         this.processDAO = factory.createProcessDAO();
      }

      return this.processDAO;
   }

   @GET
   @Path("definitions")
   @Produces("application/json")
   public Response getDefinitionsJSON(
     @Context
     HttpServletRequest request
   )
   {
      ProcessDefinitionWrapper wrapper =
        new ProcessDefinitionWrapper(getProcessDAO(request).getAllDefinitions());
      return createJsonResponse(wrapper);
   }

   @POST
   @Path("definitions/{id}/remove")
   @Produces("application/json")
   public Response removeDefinitionsJSON(
     @PathParam("id")
     long processId,
     @Context
     HttpServletRequest request
   )
   {
      ProcessDefinitionWrapper wrapper =
        new ProcessDefinitionWrapper( getProcessDAO(request).removeDefinition(processId));
      return createJsonResponse(wrapper);
   }

   @GET
   @Path("definitions/{id}/instances")
   @Produces("application/json")
   public Response getInstancesJSON(
     @PathParam("id")
     long processId,
     @Context
     HttpServletRequest request)
   {

      ProcessInstanceWrapper wrapper =
        new ProcessInstanceWrapper(getProcessDAO(request).getInstancesByProcessId(processId));
      return createJsonResponse(wrapper);
   }

   @POST
   @Path("definitions/{id}/instances/new")
   @Produces("application/json")
   public Response newInstance(
     @PathParam("id")
     long processId,
     @Context
     HttpServletRequest request)
   {

      ProcessInstance instance = getProcessDAO(request).newInstance(processId);
      return createJsonResponse(instance);
   }

   @POST
   @Path("instances/{id}/state/{next}")
   @Produces("application/json")
   public Response changeState(
     @PathParam("id")
     long instanceId,
     @PathParam("next")
     String next,
     @Context
     HttpServletRequest request)
   {
      getProcessDAO(request).changeState(instanceId, next);
      return Response.ok().build();
   }

   private Response createJsonResponse(Object wrapper)
   {
      Gson gson = GsonFactory.createInstance();
      String json = gson.toJson(wrapper);
      return Response.ok(json).type("application/json").build();
   }
}
