/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import com.google.gson.Gson;
import org.jboss.bpm.console.client.model.TaskReferenceWrapper;
import org.jboss.bpm.console.client.model.forms.FieldDef;
import org.jboss.bpm.console.client.model.forms.FormDef;
import org.jboss.bpm.console.client.model.util.MapEntry;
import org.jboss.bpm.console.server.dao.DAOFactory;
import org.jboss.bpm.console.server.dao.TaskDAO;
import org.jboss.bpm.console.server.dao.internal.JBPM3FormParser;
import org.jboss.bpm.console.server.gson.GsonFactory;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * REST server module for accessing process related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("tasks")
public class TaskManagement
{
   private TaskDAO taskDAO;

   /**
    * Lazy load the {@link org.jboss.bpm.console.server.dao.ProcessDAO}
    */
   private TaskDAO getTaskDAO(HttpServletRequest request)
   {
      if(null==this.taskDAO)
      {
         DAOFactory factory = DAOFactory.newInstance(request.getSession().getServletContext());
         this.taskDAO = factory.createTaskDAO();
      }

      return this.taskDAO;
   }

   @GET
   @Path("actor")
   @Produces("application/json")
   public Response getTasksForActor(
         @Context
         HttpServletRequest request,
         @QueryParam("actors")
         String actors
   )
   {
      List<String> actorList = new ArrayList<String>();

      StringTokenizer tok = new StringTokenizer(actors, ",");
      while(tok.hasMoreTokens())
      {
         actorList.add(tok.nextToken());
      }

      TaskReferenceWrapper wrapper =
            new TaskReferenceWrapper(getTaskDAO(request).getTasksByActors(actorList));
      return createJsonResponse(wrapper);
   }

   @GET
   @Path("forms/{processId}")
   @Produces("application/json")
   public Response getTasksForms(
         @Context
         HttpServletRequest request,
         @PathParam("processId")
         long processId
   )
   {
      Map<String,String> formMapping =
            getTaskDAO(request).getAvailableTaskForms(processId);

      List<MapEntry> entries = new ArrayList<MapEntry>();
      for(String s : formMapping.keySet())
      {
         entries.add( new MapEntry(s, formMapping.get(s)) );
      }

      return createJsonResponse( entries );
   }

   @GET
   @Path("forms/{processId}/{taskId}")
   @Produces("application/json")
   public Response getTasksFormByTaskId(
         @Context
         HttpServletRequest request,
         @PathParam("processId")
         long processId,
         @PathParam("taskId")
         long taskId

   )
   {
      byte[] form = getTaskDAO(request).getTaskFormByTaskName(processId, taskId);
      JBPM3FormParser parser = new JBPM3FormParser();
      final FormDef formDef = new FormDef("taskform-"+taskId);
      try
      {
         parser.parse( new ByteArrayInputStream(form));

         parser.visitResults(
               new JBPM3FormParser.ParseResultVisitor()
               {

                  public void onInputField(FieldDef field)
                  {
                     formDef.getFields().add(field);
                  }

                  public void onInputButton(JBPM3FormParser.InputButton btn)
                  {
                     // TODO: parse buttons
                  }
               }
         );
      }
      catch (Exception e)
      {
         throw new RuntimeException("Failed to parse form data", e);
      }

      return createJsonResponse(formDef);
   }

   private Response createJsonResponse(Object wrapper)
   {
      Gson gson = GsonFactory.createInstance();
      String json = gson.toJson(wrapper);
      return Response.ok(json).type("application/json").build();
   }
}
