package org.jbpm.job.executor;

import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.job.Job;
import org.jbpm.persistence.db.DbPersistenceService;
import org.jbpm.persistence.db.StaleObjectLogConfigurer;

public class LockMonitorThread extends Thread {

  public static final String DEFAULT_NAME = "LockMonitorThread";

  final JbpmConfiguration jbpmConfiguration;
  final int lockMonitorInterval;
  final int maxLockTime;
  final int lockBufferTime;

  volatile boolean isActive = true;

  public LockMonitorThread(JobExecutor jobExecutor) {
    this(DEFAULT_NAME, jobExecutor);
  }

  public LockMonitorThread(String name, JobExecutor jobExecutor) {
    super(name);
    jbpmConfiguration = jobExecutor.getJbpmConfiguration();
    lockMonitorInterval = jobExecutor.getLockMonitorInterval();
    maxLockTime = jobExecutor.getMaxLockTime();
    lockBufferTime = jobExecutor.getLockBufferTime();
  }

  /** @deprecated As of jBPM 3.2.6, replaced by {@link #LockMonitorThread(JobExecutor)} */
  public LockMonitorThread(JbpmConfiguration jbpmConfiguration, int lockMonitorInterval,
      int maxLockTime, int lockBufferTime) {
    this.jbpmConfiguration = jbpmConfiguration;
    this.lockMonitorInterval = lockMonitorInterval;
    this.maxLockTime = maxLockTime;
    this.lockBufferTime = lockBufferTime;
  }

  public void run() {
    while (isActive) {
      try {
        unlockOverdueJobs();
        jbpmConfiguration.getJobExecutor().ensureThreadsAreActive();
      }
      catch (RuntimeException e) {
        log.error("exception in " + getName(), e);
      }
      if (isActive) {
        try {
          sleep(lockMonitorInterval);
        }
        catch (InterruptedException e) {
          log.info(getName() + " got interrupted");
        }
      }
    }
    log.info(getName() + " leaves cyberspace");
  }

  protected void unlockOverdueJobs() {
    JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
    try {
      Date threshold = new Date(System.currentTimeMillis() - maxLockTime - lockBufferTime);
      List overdueJobs = jbpmContext.getJobSession().findJobsWithOverdueLockTime(threshold);
      for (Iterator i = overdueJobs.iterator(); i.hasNext();) {
        Job job = (Job) i.next();
        if (log.isDebugEnabled()) {
          log.debug("unlocking " + job + " owned by " + job.getLockOwner());
        }
        job.setLockOwner(null);
        job.setLockTime(null);
      }
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    catch (Error e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      try {
        jbpmContext.close();
      }
      catch (RuntimeException e) {
        if (!DbPersistenceService.isLockingException(e)) throw e;
        // if this is a locking exception, keep it quiet
        StaleObjectLogConfigurer.getStaleObjectExceptionsLog()
            .error("could not unlock overdue jobs");
      }
    }
  }

  /**
   * @deprecated As of jBPM 3.2.3, replaced by {@link #deactivate()}
   */
  public void setActive(boolean isActive) {
    if (isActive == false) deactivate();
  }

  /**
   * Indicates that this thread should stop running. Execution will cease shortly afterwards.
   */
  public void deactivate() {
    if (isActive) {
      isActive = false;
      interrupt();
    }
  }

  private static Log log = LogFactory.getLog(LockMonitorThread.class);
}
