/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.configuration;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.jbpm.JbpmException;
import org.jbpm.util.XmlUtil;
import org.w3c.dom.Element;

public class PropertyInfo implements Serializable {

  private static final long serialVersionUID = 1L;

  private String propertyName;
  private String setterMethodName;
  private ObjectInfo propertyValueInfo;

  public PropertyInfo(Element propertyElement, ObjectFactoryParser configParser) {
    // property
    if (propertyElement.hasAttribute("name")) {
      propertyName = propertyElement.getAttribute("name");
    }
    // setter method
    else if (propertyElement.hasAttribute("setter")) {
      setterMethodName = propertyElement.getAttribute("setter");
    }
    else {
      throw new JbpmException("missing name or setter attribute in property");
    }

    // propertyValueInfo
    Element propertyValueElement = XmlUtil.element(propertyElement);
    propertyValueInfo = configParser.parse(propertyValueElement);
  }

  public void injectProperty(Object object, ObjectFactoryImpl objectFactory) {
    Method setterMethod = findSetter(object.getClass());
    setterMethod.setAccessible(true);
    Object value = objectFactory.getObject(propertyValueInfo);
    try {
      setterMethod.invoke(object, new Object[] { value });
    }
    catch (IllegalAccessException e) {
      throw new JbpmException(getClass() + " has no access to " + setterMethod, e);
    }
    catch (InvocationTargetException e) {
      throw new JbpmException(setterMethod + " threw exception", e.getCause());
    }
  }

  public Method findSetter(Class clazz) {
    if (setterMethodName == null) {
      if (propertyName.startsWith("is") && propertyName.length() > 3
        && Character.isUpperCase(propertyName.charAt(2))) {
        setterMethodName = "set" + propertyName.substring(2);
      }
      else {
        setterMethodName = "set" + Character.toUpperCase(propertyName.charAt(0))
          + propertyName.substring(1);
      }
    }

    for (Class candidateClass = clazz; candidateClass != Object.class; candidateClass = candidateClass.getSuperclass()) {
      Method[] methods = candidateClass.getDeclaredMethods();
      for (int i = 0; i < methods.length; i++) {
        Method method = methods[i];
        if (method.getName().equals(setterMethodName) && method.getParameterTypes().length == 1) {
          return method;
        }
      }
    }

    throw new JbpmException("missing setter '" + setterMethodName + "' in " + clazz);
  }

  String getPropertyName() {
    return propertyName;
  }

  ObjectInfo getPropertyValueInfo() {
    return propertyValueInfo;
  }
}
