/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm;

import java.io.File;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.List;
import java.util.zip.ZipInputStream;


/** a deployment unit, containing all information in source format from which 
 * a process definition will be created and stored in the process repository.
 *  
 * <p>A deployment takes a bunch of files as input.  Files must have a name 
 * and optionally they can have a type.  During deployment, the type can be 
 * deducted from the filename automatically.  Use the <code>addXxxx</code>-methods
 * to add files to the deployment.  Use {@link #setFileType(String, String)} to 
 * set the file type programmatically.
 * </p>
 * 
 * <p>The deployment also manages a map of objects.  The deployers will typically 
 * generate those objects based on the corresponding file contents.  E.g. the
 * jpdl deployer will produce a  <code>order.jpdl.xml</code> 
 * {@link ProcessDefinition}-object from the <code>order.jpdl.xml</code> file.
 * </p>
 * 
 * <p>Typically, a deployer will save the created objects into the database.
 * </p>
 * 
 * <p>Users can also supply objects programmatically with {@link #addObject(String, Object)}. 
 * </p>
 * 
 * @author Tom Baeyens
 */
public interface Deployment extends Serializable {
  
  Deployment setName(String name);

  Deployment addResource(String resource);
  Deployment addFile(File file);
  Deployment addUrl(URL url);
  Deployment addInputStream(String name, InputStream inputStream);
  Deployment addString(String name, String string);
  Deployment addArchiveResource(String resource);
  Deployment addArchiveFile(File file);
  Deployment addArchiveUrl(URL url);
  Deployment addArchive(ZipInputStream zipInputStream);
  Deployment addDirectory(String directory);
  Deployment addDirectoryCanonical(String directory);
  Deployment addDirectory(File directory);
  Deployment addDirectoryCanonical(File directory);
  
  Deployment setFileType(String fileName, String fileType);

  Deployment deploy();

  /** the deployed process definitions. */
  List<ProcessDefinition> getProcessDefinitions();
  /** the problems encountered during deployment */
  List<Problem> getProblems(); 
}
