/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm;

import java.util.List;

import org.jbpm.task.Role;
import org.jbpm.task.Task;

/**
 * Human task management facility.
 * @author Alejandro Guizar
 */
public interface TaskService {

  /** Creates a task.
   * The returned task will be transient.
   * Use {@link #saveTask(Task)} to persist the task. */
  Task newTask();

  /** Saves the given task to persistent storage. */
  void saveTask(Task task);

  /** Retrieves the task with the given identifier from persistent storage. 
   * If no task with the given identifier exists, 
   * the call returns <code>null</code>. */
  Task getTask(String taskId);

  /** Marks the task with the given identifier as completed.
   * This operation may result in a process instance being resumed. */
  void submitTask(String taskId);

  /** Removes the task with the given identifier from persistent storage. */
  void deleteTask(String taskId);

  List<String> getCandidates(String taskId);
  List<Role> getRoles(String taskId);

  /** retrieves the personal task of the given user, which might be different 
   * then the current authenticated user.  E.g. when a manager wants to 
   * look at a subordinate's task list. */
  List<Task> getPersonalTaskList(String userId, int firstResult, int maxResults);

  /** retrieves the group task of the given user.
   * The user id will be resolved to a set of candidate identities
   * by the identity spi. */
  List<Task> getGroupTaskList(String userId, int firstResult, int maxResults);

}
