/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.api;

import java.io.Serializable;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.jbpm.api.client.ClientExecution;
import org.jbpm.api.client.ClientProcessDefinition;
import org.jbpm.api.model.ObservableElement;

/** a runtime path of execution.
 * 
 * <h3 id="state">State of an execution</h3>
 * 
 * <p>The state of an execution is either active or locked.  An active execution is either 
 * executing or waiting for an external trigger.  If an execution is not in {@link #STATE_ACTIVE},
 * then it is locked.  A locked execution is read only.  
 * </p>
 * 
 * <p>When a new execution is created, it is in {@link #STATE_ACTIVE}.  To change 
 * the state to a locked state, use {@link #lock(String)}.  
 * {@link #STATE_ACTIVE Some STATE_* constants} are provided that represent the 
 * most commonly used locked states.  But the state '...' in the picture indicates 
 * that any string can be provided as the state in the lock method.
 * </p>
 * 
 * <p>If an execution is locked, methods that change the execution will throw
 * a {@link JbpmException} and the message will reference the actual locking state.
 * {@link #fire(String, ObservableElement) Firing events}, 
 * {@link #setVariable(String, Object) updating variables}, 
 * {@link #setPriority(int) updating priority} and 
 * {@link #createComment(String) adding comments} are not considered to change an 
 * execution.  Also {@link #createProcessInstance(String) creation} and 
 * {@link #removeExecution(Execution) removal} of child executions are unchecked, 
 * which means that those methods can be invoked by external API clients and 
 * activity behaviour methods, even while the execution is in a locked state. 
 * </p>    
 * 
 * <p>Make sure that comparisons between {@link #getState()} and the 
 * {@link #STATE_ACTIVE STATE_* constants} are  
 * done with .equals and not with '==' because if executions are 
 * loaded from persistent storage, a new string is created instead 
 * of the constants.
 * </p>
 * 
 * <h3>Comments</h3>
 * 
 * @author Tom Baeyens
 */
public interface Execution extends Serializable {

  /** between {@link ClientProcessDefinition#createProcessInstance() creation of 
   * a new process instance} and the {@link ClientExecution#start() start} of that 
   * process instance.  The motivation of this state is that variables can be 
   * set programmatically on the process instance so that they can be used during 
   * initializations of variables and timers */
  String STATE_CREATED = "created";
  
  /** either executing or in a wait state waiting for a signal.
   * This is the normal state of an execution and the initial state 
   * when creating a new execution. Make sure that comparisons are 
   * done with .equals and not with '==' because if executions are 
   * loaded from persistent storage, a new string is created instead 
   * of the constants. */
  String STATE_ACTIVE = "active";
  
  /** parents with concurrent child executions are inactive.  
   * When an execution has concurrent child executions, it implies that 
   * this execution can't be active.  For example, at a fork, the parent
   * execution can wait inactively in the fork being till all the
   * child executions are joined.  Only leaves of the
   * execution tree can be active. Make sure that comparisons are 
   * done with .equals and not with '==' because if executions are 
   * loaded from persistent storage, a new string is created instead 
   * of the constants. */
  String STATE_INACTIVE = "inactive";
  
  /** this execution has ended. Make sure that comparisons are 
   * done with .equals and not with '==' because if executions are 
   * loaded from persistent storage, a new string is created instead 
   * of the constants. */
  String STATE_ENDED = "ended";
  
  /** indicates that this execution is temporary suspended with the
   * {@link #suspend()} method.  Human tasks of a suspended execution
   * shouldn't show up in people's task list and timers of suspended
   * executions shouldn't fire and the execution is locked. Make sure that comparisons are 
   * done with .equals and not with '==' because if executions are 
   * loaded from persistent storage, a new string is created instead 
   * of the constants. */
  String STATE_SUSPENDED = "suspended";
  
  /** indicates that this execution is doing an asynchronous continuation. */
  String STATE_ASYNC = "async";

  /** this execution has been cancelled before it came to completion.
   * Only history executions might expose this state.  This state can be 
   * passed in {@link ExecutionService#endProcessInstance(String, String)}.
   * Make sure that comparisons are 
   * done with .equals and not with '==' because if executions are 
   * loaded from persistent storage, a new string is created instead 
   * of the constants. */
  String STATE_CANCELLED = "cancelled";

  /** the externally given name or id of this execution. The id of a main 
   * path of execution is null.   Can be used to differentiate concurrent 
   * paths of execution e.g. the shipping and billing paths.  */
  String getName();

  /** the optional user provided business key that is unique within one 
   * process definition.  This could be for instance the order number.  
   * It's a user defined identifier for one execution within the scope of 
   * a single process definition. */
  String getKey();
  
  /** a globally unique identifier for this execution. */
  String getId();
  
  /** represents the current position in the process by indicating the 
   * name of the current activity. */
  String getActivityName();

  /** the <a href="#state">state</a> of this execution.  */
  String getState();

  /** is this a process instance */
  boolean isProcessInstance();

  /** is this execution active ? This is the inverse of {@link #isLocked()}. */
  boolean isActive();

  /** is this execution {@link #lock(String) locked} ? This is the inverse of {@link #isActive()}. */
  boolean isLocked();

  /** is this execution ended */
  boolean isEnded();

  /** is this execution suspended ? */
  boolean isSuspended();

  /** indicates low priorities with negative values and high priorities 
   * with positive values.  The default priority is 0, which means 
   * NORMAL. Other recognized named priorities are HIGHEST (2), HIGH (1), 
   * LOW (-1) and LOWEST (-2). For the rest, the user can set any other 
   * priority integer value, but then, the UI will have to display it as 
   * an integer and not the named value.*/
  int getPriority();
  
  // execution hierarchy access ///////////////////////////////////////////////
  
  /** the main path of execution in the <a href="package-summary.html#basicexecutionstructure">execution 
   * structure</a>.  Null will be returned in case this execution itself is the 
   * main execution path.  */
  Execution getProcessInstance();

  /** the parent execution in the <a href="package-summary.html#basicexecutionstructure">execution 
   * structure</a>.  Null will be returned in case this execution itself is the 
   * main execution path. */ 
  Execution getParent();

  /** the child executions in the <a href="package-summary.html#basicexecutionstructure">execution 
   * structure</a>.  Can be null and can be an empty collection. */ 
  Collection<Execution> getExecutions();
  
  /** maps child execution names to execution objects.  In case multiple executions 
   * have the same name, the first one is taken.  Can be null or can be an empty 
   * map.  The first execution without a name is also included with null as the key.
   */ 
  Map<String, Execution> getExecutionsMap();
  
  /** the child execution for the given name or null in case such execution doesn't exist. */ 
  Execution getExecution(String name);
  
  /** indicates if this execution has a child execution with the given executionName */
  boolean hasExecution(String executionName);
  
  /** search for an execution that is active and in the given activityName.
   * Returns null in case there is no such execution.
   * @see #findActiveActivityNames() */
  Execution findActiveExecutionIn(String activityName);
  
  /** get the set of all activities that are active. 
   * Returns an empty set in case there are no activities active. 
   * @see #findActiveExecutionIn(String) */
  Set<String> findActiveActivityNames();
}
