/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.api;

import org.jbpm.api.model.Comment;
import org.jbpm.api.task.Participation;
import org.jbpm.api.task.Task;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Human task management facility.
 * @author Alejandro Guizar
 * @author Heiko Braun <heiko.braun@jboss.com>
 */
public interface TaskService {

  /** Creates a task.
   * The returned task will be transient.
   * Use {@link #saveTask(Task)} to persist the task. Only 
   * after the invocation of {@link #saveTask(Task)}, the 
   * {@link Task#getDbid()} property will be initialized. */
  Task newTask();

  /** Creates a new subtask for the given task.
   * Make sure that the parent task is saved before the 
   * {@link Task#getDbid() dbid} is taken.
   * The returned task will be transient.
   * Use {@link #saveTask(Task)} to persist the task. */
  Task newTask(long parentTaskDbid);

  /** Saves the given task to persistent storage. 
   * @return the taskDbid */
  long saveTask(Task task);

  /** Retrieves the task with the given identifier from persistent storage. 
   * If no task with the given identifier exists, 
   * the call returns <code>null</code>. */
  Task getTask(long taskDbid);

  /** assigns this task to the given assignee. */
  void assignTask(long taskDbid, String userId);

  /** taking this task will prevent all other candidates from 
   * taking and working on this task. 
   * @throws JbpmException if this task already has been taken. */
  void takeTask(long taskDbid, String userId);

  /** Deletes this task, marks the related history task as completed.
   * If the task was created in the context 
   * of a process execution, this operation may result in a process instance 
   * being triggered. */
  void completeTask(long taskDbid);
  
  /** Deletes this task, marks the related history task as completed 
   * with the specified outcome.  If the task was created in the context 
   * of a process execution, this operation may result in a process instance 
   * being triggered. The outcome in that case corresponds to an outgoing 
   * transition in the process. */
  void completeTask(long taskDbid, String outcome);

  /** Deletes the task without completing it.
   * If this task was created in the context of a process execution, one 
   * of three things might happen depending on configurations of the task 
   * in the process definition: 
   * a) process executions resumes 
   * b) whole process execution is cancelled as well.
   * c) exception is thrown */
  void deleteTask(long taskDbid);

  /** Deletes the task without completing indicating the reason.  Example reasons 
   * could be: "failed", "error", "exited", "obsolete" or "deleted". 
   * If this task was created in the context of a process execution, one 
   * of three things might happen depending on configurations of the task 
   * in the process definition: 
   * a) process executions resumes 
   * b) whole process execution is cancelled as well.
   * c) exception is thrown */
  void deleteTask(long taskDbid, String reason);

  /** add a role to a given task.
   * @param participation specifies the kind of involvement of the participatingUser
   * in this task. see {@link Participation} for default constants. */
  void addTaskParticipatingUser(long taskDbid, String userId, String participationType);
    
  /** add a role to a given task.
   * @param participation specifies the kind of involvement of the participatingUser
   * in this task. see {@link Participation} for default constants. */
  void addTaskParticipatingGroup(long taskDbid, String groupId, String participationType);
    
  /** get roles related to a given task. */
  List<Participation> getTaskParticipations(long taskDbid);
  
  /** removes a role to a given task.  Nothing happens (no exception) if 
   * the role does not exist.
   * @param participation specifies the kind of involvement of the participatingUser
   * in this task. see {@link Participation} for default constants. */
  void removeTaskParticipatingUser(long taskDbid, String userId, String participationType);
  
  /** removes a role to a given task.  Nothing happens (no exception) if 
   * the role does not exist.
   * @param participation specifies the kind of involvement of the participatingUser
   * in this task. see {@link Participation} for default constants. */
  void removeTaskParticipatingGroup(long taskDbid, String groupId, String participationType);

  /** create a new query for tasks */
  TaskQuery createTaskQuery();

  /** list of tasks that are assigned to the given user. 
   * Returns an empty list in case no such tasks exist. */
  List<Task> findAssignedTasks(String userId);

  /** list of tasks that can be taken by the given user.
   * Returns an empty list in case no such tasks exist. */
  List<Task> findTakableTasks(String userId);

  /** get the subtasks for this task.  Only goes one level deep at a time. */
  List<Task> getSubTasks(long taskDbid);

  /** add a comment to a task */
  Comment addTaskComment(long taskDbid, String message);

  /** get the list of comments made to a task.  this will 
   * fetch all the comments and recursively all replies to those 
   * comments. */
  List<Comment> getTaskComments(long taskDbid);

  /** add a reply to another comment */
  Comment addReplyComment(long commentDbid, String message);

  /** delete a comment.
   * this will recursively delete all replies to this comment. */
  void deleteComment(long commentDbid);

  /** creates or overwrites a variable value on the given task */
  void setVariable(long taskDbid, String name, Object value);

  /** creates or overwrites the variable values on the given task */
  void setVariables(long taskDbid, Map<String, Object> variables);

  /** retrieves a variable */
  Object getVariable(long taskDbid, String variableName);

  /** all the variables visible in the given task */
  Set<String> getVariableNames(long taskDbid);

  /** retrieves a map of variables */
  Map<String, Object> getVariables(long taskDbid, Set<String> variableNames);

}
