/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.api.activity;

import org.jbpm.api.Execution;
import org.jbpm.api.JbpmException;
import org.jbpm.api.client.ClientProcessDefinition;
import org.jbpm.api.client.ClientProcessInstance;
import org.jbpm.api.env.Environment;
import org.jbpm.api.model.Activity;
import org.jbpm.api.model.ObservableElement;
import org.jbpm.api.model.OpenExecution;
import org.jbpm.api.model.Transition;
import org.jbpm.api.session.PvmDbSession;
import org.jbpm.api.task.Task;


/** view upon an {@link Execution path of execution} exposed to 
 * {@link ActivityBehaviour} implementations.
 *  
 * @author Tom Baeyens
 */
public interface ActivityExecution extends OpenExecution {
  
  // wait state behaviour /////////////////////////////////////////////////////
  
  /** makes this execution wait in the current activity until an external trigger is given 
   * with one of the {@link #signal()} methods. */
  void waitForSignal();

  // taking a transition //////////////////////////////////////////////////////
  
  /** takes the default transition.  
   * 
   * <p>This method can only be called from inside
   * {@link ExternalActivityBehaviour} implementations and in rare occasions also from outside 
   * of the execution (from an external client while the process is in a wait state).
   * For external clients, it is more normal to use the {@link #signal()} 
   * method as in that case, it's the current activity (hence the process language)that 
   * will decide how to interpret the signal.
   * </p>
   * 
   * @throws JbpmException in case there is no default transition in the current activity 
   * or in case this method is called from inside an {@link ActivityBehaviour} */
  void takeDefaultTransition();

  /** takes the outgoing transition with the given name. 
   * 
   * <p>This method can only be called 
   * from inside {@link ExternalActivityBehaviour} implementations and in rare occasions also from 
   * outside of the execution (from an external client while the process is in a wait state).
   * For external clients, it is more normal to use the {@link #signal(String)} 
   * method as in that case, it's the current activity (hence the process language)that 
   * will decide how to interpret the signal.</p>
   * 
   * <p>Transitions will be looked up recursively starting from the 
   * {@link #getActivity() current activity} and then up the {@link Activity#getParent() activity-parent-hierarchy} </p>
   * 
   * @param transitionName is the name of the transition to take.  A null value will 
   * match the first unnamed transition.
   *   
   * @throws JbpmException in case no such transition is found in {@link #getActivity() the current activity} 
   * or in case this method is called from inside an {@link ActivityBehaviour}.*/
  void take(String transitionName);

  /** takes the given outgoing transition.  
   * 
   * <p>This method can only be called 
   * from inside {@link ExternalActivityBehaviour} implementations and in rare occasions also from 
   * outside of the execution (from an external client while the process is in a wait state).
   * For external clients, it is more normal to use the {@link #signal(String)} 
   * method as in that case, it's the current activity (hence the process language)that 
   * will decide how to interpret the signal.</p>
   * 
   * <p>CAUTION: It's up to the client to make sure 
   * that this transition makes sense as there is no check whether the given transition 
   * is an outgoing transition of the current activity.  The motivation for that is 
   * that in case of superstates, that check can become too 'expensive'. </p> */
  void take(Transition transition);

  /** let's the given execution take the transition.
   * @throws JbpmException if the execution is not part of this process instance. */
  void take(Transition transition, Execution execution);
  
  // execute a child activity /////////////////////////////////////////////////////
  
  /** executes the given nested activity. 
   * 
   * <p>The activityName is looked up in the current activity's nested activities.</p>
   * 
   * <p>This method can only be called 
   * from inside {@link ExternalActivityBehaviour} implementations and in rare occasions also from 
   * outside of the execution (from an external client while the process is in a wait state).
   * For external clients, it is more normal to use the {@link #signal(String)} 
   * method as in that case, it's the current activity (hence the process language)that 
   * will decide how to interpret the signal.</p> */
  void execute(String activityName);
  
  /** executes the given activity. 
   * 
   * <p>This method can only be called 
   * from inside {@link ExternalActivityBehaviour} implementations and in rare occasions also from 
   * outside of the execution (from an external client while the process is in a wait state).
   * For external clients, it is more normal to use the {@link #signal(String)} 
   * method as in that case, it's the current activity (hence the process language)that 
   * will decide how to interpret the signal.</p>
   */
  void execute(Activity activity);
  
  // reposition the execution in another activity /////////////////////////////////

  /** position this execution in the destination activity. */
  void setActivity(Activity destination);

  /** position the given execution in the destination activity */
  void setActivity(Activity destination, Execution execution);
  
  // ending an execution //////////////////////////////////////////////////////
  
  /** ends this execution and all of its child executions.
   * 
   * <p>The execution will be removed from it's parent.  Potentially this can cause 
   * a parent execution to start executing in case this is the last concurrent 
   * execution for which the parent is waiting.</p> 
   */
  void end();

  /** ends this execution and all it's child executions with a user defined 
   * status.  
   * 
   * <p>It is not recommended to use any of 
   * {@link #STATE_ACTIVE the defined statuses} as that may case unpredictable 
   * side effects.</p>
   *  
   * <p>The execution will be removed from it's parent.</p> */
  void end(String state);
  
  /** ends the given execution and all it's child executions. */
  void end(OpenExecution executionToEnd);

  /** ends the given execution and all it's child executions with a user defined 
   * status. */
  void end(OpenExecution executionToEnd, String state);

  // firing events ////////////////////////////////////////////////////////////
  
  /** fires the event on the given eventSource and then propagates the event 
   * up to the eventSource's parent chain.  All the actions will see the given 
   * eventSource in {@link #getEventSource()}, event if the events are 
   * registered to parent's of the given eventSource. */
  void fire(String eventName, ObservableElement eventSource);
  
  // extra state information methods //////////////////////////////////////////
  
  /** the current transition indicating the position in the process definition graph.
   * Can be null in case this execution is not taking a transition. */
  Transition getTransition();
  
  // extensions //////////////////////////////////////////////////////////////

  /** way to access process language extensions in the execution without 
   * having to cast.  Casting can be problematic for persistence. */
  <T> T getExtension(Class<T> extensionClass);
  
  /** setter for the priority.  The default priority is 0, which means 
   * NORMAL. Other recognized named priorities are HIGHEST (2), HIGH (1), 
   * LOW (-1) and LOWEST (-2). For the rest, the user can set any other 
   * priority integer value, but then, the UI will have to display it as 
   * an integer and not the named value.*/
  void setPriority(int priority);

  
  // previous methods /////////////////////////////////////////////////////////
  
  // TODO evaluate the previous methods (JBPM-)
  // these methods are kind of performance optimisations.  a sequence and 
  // some other specific control flow implementations can be optimised if they 
  // have access to the previous activity or transition.
  // Those activities could also be implemented by letting the activities store 
  // the contextual information in process variables or some other execution 
  // context.  But with the previous properties as done now, these control flow 
  // activities can be implemented without storing an extra record.  It's only a property
  // that is only updated when the activity configuration indicates that it's needed.
  // (see also Activity.isPreviousNeeded())
  
  /** returns the previously executed activity only if {@link Activity#isPreviousNeeded()}
   * is set to true. */  
  Activity getPreviousActivity();
  
  /** returns the previously taken transition only if {@link Activity#isPreviousNeeded()}
   * is set to true. */  
  Transition getPreviousTransition();

  /** record history event that specifies for a decision activity which transition 
   * has been taken. */
  void historyDecision(String transitionName);

  /** records the end of an automatic event.  This should be called at the end of 
   * the activity.  Before the activity's execute method is invoked, the start time 
   * is automatically recorded.  And invocation of this historyAutomatic will 
   * capture the end time. */
  void historyAutomatic();

  /** marks the start of an activity for history purposes. */
  void historyActivityStart();

  /** marks the end of an activity for history purposes. */
  void historyActivityEnd();

  /** marks the end of an activity with a specific transitionName for history purposes. */
  void historyActivityEnd(String transitionName);
}
